import React from 'react';
import PropTypes from 'prop-types';
import {push} from 'connected-react-router';
import moment from 'moment';

import {Button} from '@components/Button';
import {Spin} from '@components/Spin';

import ReturnToService from '@blocks/morda/components/return_button/return_button.jsx';
import EditDisplayName from '../../edit_display_name/edit_display_name.jsx';
import {fetchPersonalInfo} from './';
import {setPersonalInfoUpdateStatus, clearPersonalInfoFieldErrors} from './actions';

import util from '../../utils';
import metrics from '../../metrics';

const ANOTHER = 'ANOTHER';

export default class DisplayNameEditForm extends React.Component {
    static getSplittedDisplayName(name) {
        const regexCodePoint = /[^\uD800-\uDFFF]|[\uD800-\uDBFF][\uDC00-\uDFFF]|[\uD800-\uDFFF]/g;
        const match = name.match(regexCodePoint);
        const result = {};

        if (match) {
            result.head = match[0];
            result.tail = match.slice(1).join('');
        }

        return result;
    }
    static getCurrentDisplayName({publicDisplayName, isDisplayNameEmpty = false, displayName = ''} = {}) {
        return isDisplayNameEmpty ? publicDisplayName : displayName;
    }

    constructor(props) {
        super(props);

        const entry = this.getSelectEntry();

        this.state = {
            select: entry,
            customName: DisplayNameEditForm.getCurrentDisplayName(props.person)
        };
        this.isFromVariants = entry !== ANOTHER;
        this.saveDisplayName = this.saveDisplayName.bind(this);
        this.onChange = this.onChange.bind(this);

        moment.locale(props.settings.language);
        this.date = moment(Date.now()).format('D MMMM YYYY');
        this.skipNextUpdates = false;
    }

    componentWillReceiveProps(nextProps) {
        const retpath = this.props.common.retpath;
        const person = nextProps.person;

        if (person.progress) {
            return;
        }

        if (nextProps.person.updated) {
            this.props.dispatch(setPersonalInfoUpdateStatus(false));

            if (typeof this.props.close === 'function') {
                this.props.close();
                return;
            }

            if (typeof window === 'undefined') {
                return;
            }

            if (retpath && window.location.href !== retpath) {
                if (util.isThirdPartyUrl(retpath)) {
                    window.location = retpath;
                    return;
                }

                this.props.dispatch(push(retpath));
                return;
            }

            this.props.dispatch(push('/profile'));
            return;
        }

        if (nextProps.common.edit === '') {
            this.skipNextUpdates = false;
        } else if (!this.skipNextUpdates) {
            this.skipNextUpdates = true;
            this.setState({
                select: this.getSelectEntry(),
                customName: DisplayNameEditForm.getCurrentDisplayName(person)
            });
        }
    }

    getSelectEntry() {
        const person = this.props.person;
        const currentDisplayName = DisplayNameEditForm.getCurrentDisplayName(person);

        for (const name in person.displayNames) {
            if (person.displayNames.hasOwnProperty(name) && name === currentDisplayName) {
                return person.displayNames[name];
            }
        }

        return ANOTHER;
    }

    getSelectKeyByValue(value) {
        const person = this.props.person;

        for (const key in person.displayNames) {
            if (person.displayNames.hasOwnProperty(key) && person.displayNames[key] === value) {
                return key;
            }
        }

        return '';
    }

    onChange({select, name}) {
        if (this.props.person.errors.display_name) {
            this.props.dispatch(clearPersonalInfoFieldErrors('display_name'));
        }

        if (select) {
            this.isFromVariants = select !== ANOTHER;
            this.setState({
                select
            });
            return;
        }

        this.isFromVariants = false;
        this.setState({
            customName: name
        });
    }

    saveDisplayName(event) {
        const name = this.state.customName.trim();

        if (event && typeof event.preventDefault === 'function') {
            event.preventDefault();
        }

        this.props.dispatch(
            fetchPersonalInfo(
                {
                    display_name: this.isFromVariants ? this.state.select : name,
                    is_from_variants: this.isFromVariants,
                    passErrors: true
                },
                true
            )
        );
        metrics.send(['Изменить персональные данные', 'Сохранить display name']);
    }

    render() {
        const {
            modal,
            person,
            dispatch,
            close,
            settings: {avatar, ua},
            common: {retpath}
        } = this.props;
        const {select, customName} = this.state;
        const {isMobile, isTouch, isTablet} = ua;
        const isPhone = (isMobile || isTouch) && !isTablet;
        const name = DisplayNameEditForm.getSplittedDisplayName(
            this.isFromVariants ? this.getSelectKeyByValue(select) : customName
        );
        const avatarPath = avatar.pathname
            .replace('%uid%', person.avatarId)
            .replace('%login%', person.login)
            .replace('%size%', '50');
        const avatarStyle = {
            background: `url("//${avatar.host}${avatarPath}") center no-repeat`
        };

        return (
            <div className={`p-control-form${modal ? ' p-control-modal dname-edit__control_modal' : ''}`}>
                <form method='post' onSubmit={this.saveDisplayName} className='dname-edit'>
                    <div className='section-title'>{i18n('Profile.personal.your-name')}</div>
                    <EditDisplayName
                        person={person}
                        dispatch={dispatch}
                        onChange={this.onChange}
                        select={select}
                        name={customName}
                    />
                    <div className='dname-edit__content'>
                        {i18n('Profile.displayname.description')}
                        <div className='dname-edit__example'>
                            <div className='dname-edit__user'>
                                <div className='dname-edit__avatar' style={avatarStyle} />
                                <div className='dname-edit__displayname'>
                                    <span>
                                        <span className='dname-edit_color_red'>{name.head}</span>
                                        <span>{name.tail}</span>
                                    </span>
                                </div>
                                <div className='dname-edit__date'>{this.date}</div>
                            </div>
                            <div className='dname-edit__example-desc'>{i18n('Profile.displayname.example')}</div>
                        </div>
                    </div>
                    <div className='p-control-saveblock dname-edit__saveblock'>
                        <div className='p-control-saveblock-cell-left'>
                            <ReturnToService retpath={retpath} isModal={modal} isMobile={isPhone} />
                        </div>
                        {modal && (
                            <div className='p-control-saveblock-cell-left'>
                                <span className='p-control-saveblock-button'>
                                    <Button size='l' view='pseudo' onClick={close}>
                                        {i18n('_AUTH_.cancel.operation')}
                                    </Button>
                                </span>
                            </div>
                        )}
                        <div className='p-control-saveblock-cell-right personal-saveblock'>
                            <span className='p-control-saveblock-button'>
                                <Button size='l' view='action' type='submit'>
                                    {i18n('Frontend.save')}
                                </Button>
                            </span>
                            <span className='personal-spinner'>
                                <Spin progress={person.progress} size='s' />
                            </span>
                        </div>
                    </div>
                </form>
            </div>
        );
    }
}

DisplayNameEditForm.propTypes = {
    modal: PropTypes.bool,
    person: PropTypes.object,
    dispatch: PropTypes.func,
    settings: PropTypes.object,
    close: PropTypes.func,
    common: PropTypes.object
};
