import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {Input} from '@components/Input';
import AutofocusInput from './autofocus_input.jsx';
import {submitName, RESTORE_LOGIN_GOAL_PREFIX} from './actions';
import metrics from '@blocks/metrics';

const errorMessages = {
    firstName: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        rateLimitExceeded: i18n('_AUTH_.question.attempts_limit_exceeded')
    },
    lastName: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        rateLimitExceeded: i18n('_AUTH_.question.attempts_limit_exceeded')
    }
};

class AskName extends PureComponent {
    constructor(props) {
        super(props);

        this.submitName = this.submitName.bind(this);
        this.onChange = this.onChange.bind(this);

        this.state = {
            lastName: ''
        };
    }

    submitName(event) {
        const firstName = this.refs.firstName.value();
        const lastName = this.state.lastName;

        event.preventDefault();
        this.props.dispatch(submitName({firstName, lastName}));
    }

    onChange(event) {
        const {target = {}} = event;
        const {value} = target;

        this.setState({lastName: value});
    }

    componentDidMount() {
        metrics.send(`${RESTORE_LOGIN_GOAL_PREFIX}_open_ask_name`);
        metrics.goal(`${RESTORE_LOGIN_GOAL_PREFIX}_open_ask_name`);
    }

    render() {
        const {errors, firstName, lastName, loading} = this.props;

        return (
            <div>
                <header className='login-restore__header'>
                    <h2 className='login-restore__title'>{i18n('Profile.restore.login.restore-title')}</h2>
                    <h3 className='login-restore__subtitle'>{i18n('Profile.restore.login.name-description')}</h3>
                </header>
                {errors.firstName === 'notMatched' ? (
                    <div>
                        <strong className='login-restore__label login-restore__label_padding'>
                            {i18n('Profile.restore.login.name-error')}
                        </strong>
                        <div className='login-restore__error-block clearfix'>
                            <div className='login-restore__error-block-phone'>{`${firstName} ${lastName}`}</div>
                            <div className='login-restore__error-block-button'>
                                <Button
                                    onClick={this.props.clearErrors}
                                    size='l'
                                    view='pseudo'
                                    type='submit'
                                    text={i18n('Profile.common.one-more-time')}
                                />
                            </div>
                        </div>
                    </div>
                ) : (
                    <form onSubmit={this.submitName}>
                        <div className='login-restore__input_name' data-t='firstname-field'>
                            <label className='login-restore__label'>{i18n('_AUTH_.field_firstname')}</label>

                            <AutofocusInput ref='firstName' name='firstName' />

                            {Boolean(errors.firstName) && (
                                <div className='p-control-error-block'>{errorMessages.firstName[errors.firstName]}</div>
                            )}
                        </div>
                        <div className='login-restore__input_name' data-t='lastname-field'>
                            <label className='login-restore__label'>{i18n('_AUTH_.field_lastname')}</label>

                            <Input
                                size='l'
                                ref='lastName'
                                value={this.state.lastName}
                                onChange={this.onChange}
                                name='lastName'
                            />

                            {Boolean(errors.lastName) && (
                                <div className='p-control-error-block'>{errorMessages.lastName[errors.lastName]}</div>
                            )}
                        </div>
                        <div className='p-control-saveblock'>
                            <div
                                className='p-control-saveblock-cell-right p-control-saveblock-button'
                                data-metrics='no_account_from_name'
                                onMouseDown={this.props.sendMetrics}
                            >
                                {Boolean(loading) && (
                                    <Button
                                        size='l'
                                        view='action'
                                        type='submit'
                                        disabled={true}
                                        text={i18n('Profile.common.continue')}
                                    />
                                )}

                                {Boolean(!loading) && (
                                    <Button
                                        size='l'
                                        view='action'
                                        type='submit'
                                        text={i18n('Profile.common.continue')}
                                    />
                                )}
                            </div>
                        </div>
                    </form>
                )}
            </div>
        );
    }
}

AskName.propTypes = {
    dispatch: PropTypes.func.isRequired,
    clearErrors: PropTypes.func.isRequired,
    sendMetrics: PropTypes.func.isRequired,
    errors: PropTypes.object.isRequired,
    firstName: PropTypes.string,
    lastName: PropTypes.string,
    loading: PropTypes.bool.isRequired
};

export default AskName;
