import React, {Component, PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {Input} from '@components/Input';
import {Captcha} from '@components/Captcha';
import AutofocusInput from './autofocus_input.jsx';
import {submitPhone, goBack, RESTORE_LOGIN_GOAL_PREFIX} from './actions';
import metrics from '@blocks/metrics';

const errorMessages = {
    phone: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        invalid: i18n('_AUTH_.phones.errors.phone.invalid'),
        smsLimitExceeded: i18n('_AUTH_.phones.errors.limit-exceeded'),
        rateLimitExceeded: i18n('_AUTH_.phone-confirm_errors_confirmations_limitexceeded_one_number')
    },
    captcha: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        notMatched: i18n('Profile.domik.captcha-not-matched')
    }
};

class AskPhone extends PureComponent {
    constructor(props) {
        super(props);

        this.submitPhone = this.submitPhone.bind(this);
        this.onChange = this.onChange.bind(this);

        this.state = {
            captcha: ''
        };
    }

    componentDidMount() {
        metrics.send(`${RESTORE_LOGIN_GOAL_PREFIX}_open_ask_phone`);
        metrics.goal(`${RESTORE_LOGIN_GOAL_PREFIX}_open_ask_phone`);
    }

    submitPhone(event) {
        const phone = this.refs.phone.value();
        const captcha = this.state.captcha;

        event.preventDefault();
        this.props.dispatch(submitPhone({phone, captcha}));
    }

    onChange(event) {
        const {target = {}} = event;
        const {value} = target;

        this.setState({captcha: value});
    }

    goBack() {
        this.props.dispatch(goBack());
    }

    render() {
        const {errors, isMobileCaptcha, tld, language, phone, captcha, dispatch, loading, step} = this.props;

        return (
            <div>
                <header className='login-restore__header'>
                    <h2 className='login-restore__title'>{i18n('Profile.restore.login.restore-title')}</h2>
                    <h3 className='login-restore__subtitle'>{i18n('Profile.restore.login.phone-label')}</h3>
                </header>
                {errors.phone === 'notFound' ? (
                    <ErrorBlock phone={phone} clearErrors={this.props.clearErrors} tld={tld} />
                ) : (
                    <form onSubmit={this.submitPhone}>
                        <label className='login-restore__label'>
                            {i18n('Profile.personal.hangover.single_phone')}:
                        </label>
                        <div className='login-restore__input login-restore__input_phone'>
                            <AutofocusInput ref='phone' name='phone' />

                            {Boolean(errors.phone) && (
                                <div className='p-control-error-block'>{errorMessages.phone[errors.phone]}</div>
                            )}
                        </div>
                        <label className='login-restore__label'>{i18n('Profile.restore.login.code-label')}</label>
                        <div className='login-restore__input login-restore__input_captcha'>
                            <Input
                                ref='captcha'
                                value={this.state.captcha}
                                onChange={this.onChange}
                                name='captcha'
                                size='l'
                            />

                            {Boolean(errors.captcha) && (
                                <div className='p-control-error-block'>{errorMessages.captcha[errors.captcha]}</div>
                            )}
                        </div>
                        <div className='login-restore__captcha'>
                            <Captcha
                                isMobile={isMobileCaptcha}
                                lang={language}
                                introSound={captcha.introSound}
                                captchaSound={captcha.captchaSound}
                                imageUrl={captcha.imageUrl}
                                loading={captcha.loading}
                                loadingAudio={captcha.loadingAudio}
                                playing={captcha.playing}
                                type={captcha.type}
                                dispatch={dispatch}
                                visible={true}
                            />
                        </div>
                        <div className='p-control-saveblock'>
                            {Boolean(step.length > 2) && (
                                <div className='p-control-saveblock-cell-left p-control-saveblock-button'>
                                    <Button
                                        view='pseudo'
                                        size='l'
                                        onClick={this.goBack}
                                        text={i18n('Profile.common.get-back')}
                                    />
                                </div>
                            )}

                            <div className='p-control-saveblock-cell-right p-control-saveblock-button'>
                                {Boolean(loading) && (
                                    <Button
                                        size='l'
                                        view='action'
                                        type='submit'
                                        disabled={true}
                                        text={i18n('Profile.common.continue')}
                                    />
                                )}

                                {Boolean(!loading) && (
                                    <Button
                                        size='l'
                                        view='action'
                                        type='submit'
                                        text={i18n('Profile.common.continue')}
                                    />
                                )}
                            </div>
                        </div>
                    </form>
                )}
            </div>
        );
    }
}

class ErrorBlock extends Component {
    shouldComponentUpdate() {
        return false;
    }

    componentDidMount() {
        const SUPPORT_DOMAINS = ['ru', 'ua', 'com', 'com.tr'];
        const EN_SUPPORT_DOMAINS = ['fr', 'co.il'];
        const tld = SUPPORT_DOMAINS.find((el) => el === this.props.tld)
            ? this.props.tld
            : EN_SUPPORT_DOMAINS.find((el) => el === this.props.tld)
            ? 'com'
            : 'ru';
        const supportURL = `https://yandex.${tld}/support/passport/troubleshooting/forgot-login-email.html`;
        const linkClasses = 'control link link_theme_normal';
        const beginOfLink = `<a class='${linkClasses}' target='_blank' href='${supportURL}'>`;
        const endOfLink = '</a>';
        const message = i18n('Profile.restore.login.support-link', beginOfLink, endOfLink);

        this.refs.supportLink.innerHTML = message;
        metrics.send(`${RESTORE_LOGIN_GOAL_PREFIX}_phone_not_found`);
        metrics.goal(`${RESTORE_LOGIN_GOAL_PREFIX}_phone_not_found`);
    }

    render() {
        const {phone} = this.props;

        return (
            <div>
                <strong className='login-restore__label login-restore__label_padding'>
                    {i18n('Profile.restore.login.phone-error')}
                </strong>
                <div className='login-restore__error-block clearfix'>
                    <div className='login-restore__error-block-phone'>{phone}</div>
                    <div className='login-restore__error-block-button'>
                        <Button
                            onClick={this.props.clearErrors}
                            size='l'
                            view='pseudo'
                            type='submit'
                            text={i18n('Profile.restore.login.try-another')}
                        />
                    </div>
                </div>
                <div className='login-restore__error-block-support' ref='supportLink' />
            </div>
        );
    }
}

ErrorBlock.propTypes = {
    phone: PropTypes.string.isRequired,
    tld: PropTypes.string.isRequired,
    clearErrors: PropTypes.func.isRequired
};

AskPhone.propTypes = {
    dispatch: PropTypes.func.isRequired,
    clearErrors: PropTypes.func.isRequired,
    phone: PropTypes.string,
    isMobileCaptcha: PropTypes.bool.isRequired,
    errors: PropTypes.object.isRequired,
    captcha: PropTypes.object.isRequired,
    loading: PropTypes.bool.isRequired,
    step: PropTypes.arrayOf(PropTypes.string).isRequired,
    language: PropTypes.string.isRequired,
    tld: PropTypes.string.isRequired
};

export default AskPhone;
