import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import SuggestedLogins from './suggested_logins.jsx';
import AskPhone from './ask_phone.jsx';
import VerifyCode from './verify_code.jsx';
import AskName from './ask_name.jsx';
import {changeLoginRecoveryStep, RESTORE_LOGIN_GOAL_PREFIX, clearErrors} from './actions';
import metrics from '@blocks/metrics';

class RestoreLogin extends Component {
    static mapStateToProps(state) {
        const {restorationLoginUrl, authUrl, addUserUrl, restorationUrl} = state.common;
        const {ua, avatar, language} = state.settings;
        const {
            step,
            phone,
            firstName,
            lastName,
            errors,
            countdown,
            suggestedLogins,
            loading,
            selectedLogin
        } = state.restoreLogin;
        const isMobile = Boolean((ua.isMobile || ua.isTouch) && !ua.isTablet);
        const isMobileCaptcha = Boolean(ua.isMobile || ua.isTablet);
        const tld = state.settings.tld;

        return {
            tld,
            isMobile,
            isMobileCaptcha,
            captcha: state.captcha,
            language,
            firstName,
            lastName,
            countdown,
            loading,
            step,
            phone,
            captchaValue: state.restoreLogin.captchaValue,
            errors,
            // eslint-disable-next-line no-useless-escape
            suggestedLogins: suggestedLogins.filter((item) => !item.login.match(/^uid\-/)),
            selectedLogin,
            avatarHost: avatar.host,
            avatarPath: avatar.pathname,
            restorationLoginUrl,
            authUrl,
            addUserUrl,
            restorationUrl
        };
    }

    constructor(props) {
        super(props);

        this.clearErrors = this.clearErrors.bind(this);
        this.sendMetrics = this.sendMetrics.bind(this);
    }

    clearErrors() {
        this.props.dispatch(clearErrors());
    }

    sendMetrics(event) {
        const visitParams = event.currentTarget.dataset.metrics;

        if (visitParams) {
            metrics.send(`${RESTORE_LOGIN_GOAL_PREFIX}_${visitParams}`);
            metrics.goal(`${RESTORE_LOGIN_GOAL_PREFIX}_${visitParams}`);
        }
    }

    componentWillMount() {
        if (this.props.suggestedLogins.length === 0) {
            this.props.dispatch(changeLoginRecoveryStep('ask_phone'));
        }
    }

    render() {
        const {
            step,
            phone,
            tld,
            errors,
            loading,
            suggestedLogins,
            firstName,
            lastName,
            countdown,
            isMobile,
            isMobileCaptcha,
            language,
            captcha,
            dispatch,
            avatarHost,
            avatarPath,
            selectedLogin,
            restorationLoginUrl,
            authUrl,
            addUserUrl,
            restorationUrl
        } = this.props;
        const lastStep = step[step.length - 1];

        return (
            <div className='login-restore__wrapper'>
                <section className={`login-restore login-restore_${lastStep}`}>
                    {(() => {
                        if (lastStep === 'suggested_logins' || lastStep === 'name_suggested_logins') {
                            return (
                                <SuggestedLogins
                                    isMobile={isMobile}
                                    selectedLogin={selectedLogin}
                                    sendMetrics={this.sendMetrics}
                                    dispatch={dispatch}
                                    avatarHost={avatarHost}
                                    avatarPath={avatarPath}
                                    suggestedLogins={suggestedLogins}
                                    restorationLoginUrl={restorationLoginUrl}
                                    authUrl={authUrl}
                                    addUserUrl={addUserUrl}
                                    restorationUrl={restorationUrl}
                                    lastStep={lastStep}
                                />
                            );
                        }

                        if (lastStep === 'ask_phone') {
                            return (
                                <AskPhone
                                    tld={tld}
                                    step={step}
                                    phone={phone}
                                    language={language}
                                    isMobileCaptcha={isMobileCaptcha}
                                    errors={errors}
                                    dispatch={dispatch}
                                    loading={loading}
                                    clearErrors={this.clearErrors}
                                    captcha={captcha}
                                />
                            );
                        }

                        if (lastStep === 'verify_code') {
                            return (
                                <VerifyCode
                                    dispatch={dispatch}
                                    phone={phone}
                                    errors={errors}
                                    countdown={countdown}
                                    loading={loading}
                                />
                            );
                        }

                        if (lastStep === 'ask_name') {
                            return (
                                <AskName
                                    dispatch={dispatch}
                                    clearErrors={this.clearErrors}
                                    sendMetrics={this.sendMetrics}
                                    errors={errors}
                                    loading={loading}
                                    firstName={firstName}
                                    lastName={lastName}
                                />
                            );
                        }

                        return null;
                    })()}
                </section>
            </div>
        );
    }
}

RestoreLogin.propTypes = {
    tld: PropTypes.string.isRequired,
    restorationLoginUrl: PropTypes.string.isRequired,
    authUrl: PropTypes.string.isRequired,
    addUserUrl: PropTypes.string.isRequired,
    restorationUrl: PropTypes.string.isRequired,
    dispatch: PropTypes.func.isRequired,
    step: PropTypes.arrayOf(PropTypes.string).isRequired,
    suggestedLogins: PropTypes.array.isRequired,
    selectedLogin: PropTypes.string,
    isMobile: PropTypes.bool.isRequired,
    isMobileCaptcha: PropTypes.bool.isRequired,
    language: PropTypes.string.isRequired,
    avatarHost: PropTypes.string,
    avatarPath: PropTypes.string,
    loading: PropTypes.bool.isRequired,
    countdown: PropTypes.number,
    phone: PropTypes.string,
    code: PropTypes.string,
    captcha: PropTypes.object.isRequired,
    captchaValue: PropTypes.string,
    firstName: PropTypes.string,
    lastName: PropTypes.string,
    errors: PropTypes.object
};

const WrappedRestoreLogin = connect(RestoreLogin.mapStateToProps)(RestoreLogin);

export default WrappedRestoreLogin;
