import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Link} from '@components/Link';
import {changeLoginRecoveryStep, toggleLoginSelection, RESTORE_LOGIN_GOAL_PREFIX} from './actions';
import metrics from '@blocks/metrics';

class SuggestedLogins extends PureComponent {
    constructor(props) {
        super(props);

        this.askPhoneNumber = this.askPhoneNumber.bind(this);
        this.toggleLoginSelection = this.toggleLoginSelection.bind(this);
    }

    askPhoneNumber() {
        this.props.dispatch(changeLoginRecoveryStep('ask_phone'));
    }

    getAvatarPath(id, retina) {
        const {avatarHost, avatarPath} = this.props;
        const path = avatarPath.replace('%uid%', id).replace('%size%', retina ? 'retina-50' : '50');

        return `https://${avatarHost}${path}`;
    }

    toggleLoginSelection(event) {
        const dispatch = this.props.dispatch;
        const login = event.currentTarget.getAttribute('data-login');

        if (!this.props.isMobile) {
            return;
        }
        dispatch(toggleLoginSelection(login));
    }

    componentDidMount() {
        const step = this.props.lastStep;

        metrics.send(`${RESTORE_LOGIN_GOAL_PREFIX}_open_${step}`);
        metrics.goal(`${RESTORE_LOGIN_GOAL_PREFIX}_open_${step}`);
    }

    render() {
        const {
            lastStep,
            suggestedLogins,
            selectedLogin,
            restorationLoginUrl,
            authUrl,
            addUserUrl,
            restorationUrl
        } = this.props;

        return (
            <div>
                <header className='login-restore__header'>
                    <h2 className='login-restore__title'>{i18n('Profile.restore.login.restore-title')}</h2>
                    <h3 className='login-restore__subtitle'>
                        {Boolean(lastStep === 'name_suggested_logins') &&
                            i18n('Profile.restore.login.name-suggested-logins-description')}
                        {Boolean(lastStep === 'suggested_logins') && i18n('Profile.restore.login.phone-description')}
                    </h3>
                </header>
                <figure>
                    <figcaption className='login-restore__suggested-login-label'>
                        {Boolean(lastStep === 'name_suggested_logins') &&
                            i18n('Profile.restore.login.name-suggested-login-label')}
                        {Boolean(lastStep === 'suggested_logins') &&
                            i18n('Profile.restore.login.suggested-logins-label')}
                    </figcaption>
                    <ul className='login-restore__suggested-login-list'>
                        {suggestedLogins.map((item) => (
                            <li
                                key={item.login}
                                data-login={item.login}
                                style={{background: selectedLogin === item.login ? '#FFEBA0' : ''}}
                                onClick={this.toggleLoginSelection}
                            >
                                <div className='login-restore__suggested-login-item'>
                                    <div className='icon'>
                                        <img
                                            src={this.getAvatarPath(item.default_avatar)}
                                            width='50'
                                            height='50'
                                            srcSet={`${this.getAvatarPath(item.default_avatar, true)} 2x`}
                                        />
                                    </div>
                                    <span className='login-restore__suggested-login'>{item.login}</span>
                                    <div
                                        className='controls'
                                        style={{display: selectedLogin === item.login ? 'block' : ''}}
                                    >
                                        <span className='link'>
                                            <Link
                                                data-metrics='suggest_from_restore'
                                                onMouseDown={this.props.sendMetrics}
                                                href={addUserUrl.replace('login=login', `login=${item.login}`)}
                                            >
                                                {i18n('Frontend.logon')}
                                            </Link>
                                        </span>
                                        <span className='link'>
                                            <Link
                                                data-metrics='suggest_from_restore'
                                                onMouseDown={this.props.sendMetrics}
                                                href={restorationUrl.replace('login=login', `login=${item.login}`)}
                                            >
                                                {i18n('Profile.domik.remember-password')}
                                            </Link>
                                        </span>
                                    </div>
                                </div>
                            </li>
                        ))}
                    </ul>
                </figure>

                {Boolean(lastStep === 'name_suggested_logins') && (
                    <div>
                        <a
                            href={restorationLoginUrl}
                            className='login-restore__suggested-login-item login-restore__suggested-login-item_search'
                        >
                            <div className='icon' />
                            <span>{i18n('Profile.restore.login.retry-search')}</span>
                        </a>
                        <a
                            data-metrics='no_account_from_restore_name'
                            onMouseDown={this.props.sendMetrics}
                            href={authUrl}
                            className='login-restore__suggested-login-item
                        login-restore__suggested-login-item_no-logins'
                        >
                            <div className='icon' />
                            <span>{i18n('Profile.restore.login.signin-in-another')}</span>
                        </a>
                    </div>
                )}

                {Boolean(lastStep === 'suggested_logins') && (
                    <div
                        data-metrics='no_account_from_restore'
                        onMouseDown={this.props.sendMetrics}
                        className='login-restore__suggested-login-item
                            login-restore__suggested-login-item_no-logins'
                        onClick={this.askPhoneNumber}
                    >
                        <div className='icon' />
                        <span>{i18n('Profile.restore.login.no-suggested-login')}</span>
                    </div>
                )}
            </div>
        );
    }
}

SuggestedLogins.propTypes = {
    dispatch: PropTypes.func.isRequired,
    sendMetrics: PropTypes.func.isRequired,
    lastStep: PropTypes.string.isRequired,
    restorationLoginUrl: PropTypes.string.isRequired,
    authUrl: PropTypes.string.isRequired,
    addUserUrl: PropTypes.string.isRequired,
    restorationUrl: PropTypes.string.isRequired,
    avatarHost: PropTypes.string.isRequired,
    avatarPath: PropTypes.string.isRequired,
    suggestedLogins: PropTypes.arrayOf(
        PropTypes.shape({
            login: PropTypes.string.isRequired,
            default_avatar: PropTypes.string.isRequired
        })
    ),
    isMobile: PropTypes.bool,
    selectedLogin: PropTypes.string
};

export default SuggestedLogins;
