import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {Button} from '@components/Button';
import {Input} from '@components/Input';
import {Link} from '@components/Link';
import {submitPhone, submitCode, goBack, RESTORE_LOGIN_GOAL_PREFIX} from './actions';
import metrics from '@blocks/metrics';

const errorMessages = {
    code: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        notMatched: i18n('_AUTH_.phones.errors.code.invalid'),
        smsLimitExceeded: i18n('_AUTH_.phones.errors.limit-exceeded'),
        rateLimitExceeded: i18n('_AUTH_.phones.errors.confirmation-limit-exceeded')
    }
};

class VerifyCode extends PureComponent {
    constructor(props) {
        super(props);

        this.submitCode = this.submitCode.bind(this);
        this.resendCode = this.resendCode.bind(this);
        this.goBack = this.goBack.bind(this);
        this.onChange = this.onChange.bind(this);

        this.state = {
            code: ''
        };
    }

    submitCode(event) {
        const code = this.state.code;

        event.preventDefault();
        this.props.dispatch(submitCode({code}));
    }

    resendCode() {
        const phone = this.props.phone;

        this.props.dispatch(submitPhone({phone, resend: true}));
    }

    goBack() {
        this.props.dispatch(goBack());
    }

    onChange(event) {
        const {target = {}} = event;
        const {value} = target;

        this.setState({code: value});
    }

    componentDidMount() {
        metrics.send(`${RESTORE_LOGIN_GOAL_PREFIX}_open_verify_code`);
        metrics.goal(`${RESTORE_LOGIN_GOAL_PREFIX}_open_verify_code`);
    }

    render() {
        const {phone, errors, countdown, loading} = this.props;

        return (
            <div>
                <header className='login-restore__header'>
                    <h2 className='login-restore__title'>{i18n('Profile.restore.login.restore-title')}</h2>
                    <h3 className='login-restore__subtitle'>{i18n('Profile.restore.login.confirm-description')}</h3>
                </header>
                <form onSubmit={this.submitCode}>
                    <strong className='login-restore__label'>
                        {i18n('Profile.restore.login.confirm-label', phone)}
                    </strong>
                    <div className='clearfix'>
                        <div className='login-restore__input'>
                            <Input
                                value={this.state.code}
                                onChange={this.onChange}
                                focused={true}
                                name='code'
                                size='l'
                            />

                            {Boolean(errors.code) && (
                                <div className='p-control-error-block'>{errorMessages.code[errors.code]}</div>
                            )}
                        </div>
                        <div
                            className={classnames('login-restore__phone-controls', {
                                'login-restore__phone-controls_disabled': countdown > 0
                            })}
                        >
                            {Boolean(countdown > 0) && (
                                <div>
                                    <span>{i18n('Profile.phones.change-number')}</span>
                                    <span>{i18n('Profile.restore.login.send-again')}</span>
                                </div>
                            )}

                            {Boolean(countdown === 0) && (
                                <div>
                                    <Link onClick={this.goBack} pseudo={true}>
                                        {i18n('Profile.phones.change-number')}
                                    </Link>
                                    <Link onClick={this.resendCode} pseudo={true}>
                                        {i18n('Profile.restore.login.send-again')}
                                    </Link>
                                </div>
                            )}
                        </div>

                        {Boolean(countdown > 0) && (
                            <div className='login-restore__countdown'>
                                0:{String(countdown).replace(/^(\d)$/, '0$1')}
                            </div>
                        )}

                        <div className='p-control-saveblock-button'>
                            <Button
                                type='submit'
                                view='action'
                                size='l'
                                disabled={Boolean(loading)}
                                text={i18n('Profile.common.continue')}
                            />
                        </div>
                    </div>
                </form>
            </div>
        );
    }
}

VerifyCode.propTypes = {
    dispatch: PropTypes.func.isRequired,
    phone: PropTypes.string.isRequired,
    errors: PropTypes.object.isRequired,
    countdown: PropTypes.number.isRequired,
    loading: PropTypes.bool.isRequired
};

export default VerifyCode;
