import React, {Component} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {Button} from '@components/Button';
import {Spin} from '@components/Spin';
import {Input} from '@components/Input';
import {Select} from '@components/Select';
import ReturnToService from '@blocks/morda/components/return_button/return_button.jsx';
import {push} from 'connected-react-router';
import util from '@blocks/utils';
import metrics from '@blocks/metrics';
import {MAIN_PAGE_URL} from '@blocks/constants';
import getQuestions from './';
import {saveKvKo, showKvKo, changeQuestion, validateForm, setKvKoUpdateStatus, deleteFieldError} from './actions';

const errorMessages = {
    newQuestionId: {
        undefined: i18n('Profile.kvko.choose-question'),
        empty: i18n('Profile.kvko.choose-question')
    },
    ownQuestion: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        long: i18n('_AUTH_.hint_question_errors_toolong')
    },
    newAnswer: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        long: i18n('_AUTH_.hint_answer_errors_toolong')
    },
    oldAnswer: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        notMatched: i18n('Profile.kvko.answer-not-matched'),
        long: i18n('_AUTH_.hint_answer_errors_toolong')
    },
    rateLimit: i18n('Errors.kvko.rate_limit.v2')
        .replace('%1', '<a href="/restoration/changehint" target="_blank" class="control link link_theme_normal">')
        .replace('%2', '</a>')
};

export default class KvKoModal extends Component {
    constructor(props) {
        super(props);
        this.saveKvKo = this.saveKvKo.bind(this);
        this.selectQuestion = this.selectQuestion.bind(this);
        this.selectQuestionNativeSelect = this.selectQuestionNativeSelect.bind(this);
        this.closeKvKoModal = this.closeKvKoModal.bind(this);
        this.handleInput = this.handleInput.bind(this);

        this.state = {
            currentAnswer: '',
            updatedAnswer: '',
            ownQuestion: ''
        };
    }

    componentDidMount() {
        const {settings, security} = this.props;
        const controlQuestion = (security && security.controlQuestion) || {};

        this.isPhone = (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;

        if (typeof window !== 'undefined' && controlQuestion.available && controlQuestion.available.length === 0) {
            this.props.dispatch(getQuestions());
        }
    }

    handleInput(event) {
        const {target = {}} = event;
        const {value, name} = target;
        const keyMaps = {
            updatedAnswer: 'newAnswer',
            currentAnswer: 'oldAnswer'
        };
        const key = keyMaps[name] || name;

        if ((this.props.security.controlQuestion.errors || {})[key]) {
            this.props.dispatch(deleteFieldError(key));
        }

        this.setState({
            [name]: value
        });
    }

    saveKvKo(e) {
        const data = this.getFieldValues();

        e.preventDefault();
        this.props.dispatch(validateForm(data));
        this.props.dispatch(saveKvKo(data));

        metrics.send(['Изменить контрольный вопрос/ответ', 'Сохранить']);
    }

    setQuestionId(value) {
        this.props.dispatch(changeQuestion(value));
    }

    getFieldValues() {
        const {controlQuestion} = this.props.security;
        const newQuestionId = controlQuestion.selected;
        const data = {
            newQuestionId: controlQuestion.selected,
            newAnswer: this.state.updatedAnswer
        };

        if (newQuestionId === '99') {
            data.ownQuestion = this.state.ownQuestion;
        }

        if (controlQuestion.current) {
            data.oldAnswer = this.state.currentAnswer;
        }

        return data;
    }

    closeKvKoModal() {
        this.props.dispatch(showKvKo(false));
        this.clearForm();
        metrics.send(['Изменить контрольный вопрос/ответ', 'Отменить']);
    }

    selectQuestion(event) {
        const value = event.target && event.target.value;

        if (value) {
            this.props.dispatch(changeQuestion(value));
            if (value !== '0' && (this.props.security.controlQuestion.errors || {}).newQuestionId) {
                this.props.dispatch(deleteFieldError('newQuestionId'));
            }
        } else {
            this.props.dispatch(changeQuestion(0));
        }
    }

    selectQuestionNativeSelect(event) {
        if (event.target && event.target.value) {
            this.props.dispatch(changeQuestion(event.target.value));
            if (event.target.value !== '0' && (this.props.security.controlQuestion.errors || {}).newQuestionId) {
                this.props.dispatch(deleteFieldError('newQuestionId'));
            }
        }
    }

    clearForm() {
        this.setQuestionId('0');
        this.setState({
            currentAnswer: '',
            updatedAnswer: '',
            ownQuestion: ''
        });
    }

    componentWillReceiveProps(nextProps) {
        const props = this.props;
        const retpath = props.common.retpath || '';

        if (nextProps.security.controlQuestion.updated) {
            props.dispatch(setKvKoUpdateStatus(false));
            this.clearForm();

            if (typeof window !== 'undefined') {
                if (retpath && window.location.href !== retpath) {
                    if (util.isThirdPartyUrl(retpath)) {
                        window.location = retpath;
                        return;
                    }

                    props.dispatch(push(retpath));
                    return;
                }

                props.dispatch(push(MAIN_PAGE_URL));
                return;
            }
        }
    }

    render() {
        const {settings = {}, security = {}, common = {}} = this.props;
        const ua = settings.ua;
        const controlQuestion = security.controlQuestion;
        const {selected, available, current, errors, loading} = controlQuestion;
        const isMobile = ua.isMobile && !ua.isTablet;
        const retpath = common.retpath;

        return (
            <form
                className={classnames('p-control-form', {'p-control-modal': this.props.modal})}
                onSubmit={this.saveKvKo}
                data-t='user-question'
            >
                <div className='u-q-modal__top'>
                    <h2 className='section-title'>{i18n('Profile.kvko.title')}</h2>
                    {Boolean(available.length) && (
                        <div className='u-q-modal__field'>
                            <label className='u-q-modal__label'>{`${i18n('Profile.kvko.control-question')}:`}</label>
                            <div>
                                <div data-t='user-question-all'>
                                    <Select
                                        width='max'
                                        size='l'
                                        name='control_question'
                                        id='control_question'
                                        dataT='control_question'
                                        value={selected}
                                        onChange={this.selectQuestion}
                                        options={available.map((question) => ({
                                            value: question.val,
                                            content: question.text
                                        }))}
                                    />
                                </div>

                                {Boolean(errors.newQuestionId) && (
                                    <div className='p-control-error-block'>
                                        {errorMessages.newQuestionId[errors.newQuestionId]}
                                    </div>
                                )}
                            </div>
                        </div>
                    )}

                    <div className={classnames('u-q-modal__field', {'g-hide': selected !== '99'})}>
                        <label className='u-q-modal__label'>{`${i18n('Profile.kvko.your-question')}:`}</label>
                        <div>
                            <span data-t='user-question-own'>
                                <Input
                                    size='l'
                                    maxLength={37}
                                    name='ownQuestion'
                                    value={this.state.ownQuestion}
                                    onChange={this.handleInput}
                                />
                            </span>
                            {Boolean(errors.ownQuestion) && (
                                <div className='p-control-error-block'>
                                    {errorMessages.ownQuestion[errors.ownQuestion]}
                                </div>
                            )}
                        </div>
                    </div>
                    <div className='u-q-modal__field'>
                        <label className='u-q-modal__label'>{`${i18n('Profile.kvko.your-answer')}:`}</label>
                        <div>
                            <span data-t='user-question-new'>
                                <Input
                                    size='l'
                                    maxLength={30}
                                    name='updatedAnswer'
                                    value={this.state.updatedAnswer}
                                    onChange={this.handleInput}
                                />
                            </span>
                            {Boolean(errors.newAnswer) && (
                                <div className='p-control-error-block'>{errorMessages.newAnswer[errors.newAnswer]}</div>
                            )}
                        </div>
                    </div>
                </div>
                <div className={classnames('u-q-modal__prev-answer', {'g-hide': !current})}>
                    <label className='u-q-modal__label'>
                        <div className='u-q-modal__prev-answer-label'>{i18n('Profile.kvko.previous-answer')}</div>
                        <div className='u-q-modal__prev-answer-text'>{`"${current && current.text}":`}</div>
                    </label>
                    <div>
                        <span data-t='user-question-old'>
                            <Input
                                size='l'
                                maxLength={1024}
                                name='currentAnswer'
                                value={this.state.currentAnswer}
                                onChange={this.handleInput}
                            />
                        </span>
                        {Boolean(errors.oldAnswer) && (
                            <div className='p-control-error-block'>{errorMessages.oldAnswer[errors.oldAnswer]}</div>
                        )}
                    </div>
                    {Boolean(errors.rateLimitExceeded) && (
                        <div
                            className='u-q-modal__rate-limit-error'
                            dangerouslySetInnerHTML={{__html: errorMessages.rateLimit}}
                        />
                    )}
                </div>
                <div className='u-q-modal__field clearfix'>
                    {Boolean(this.props.modal) && (
                        <div className='p-control-saveblock-cell-left p-control-saveblock-button'>
                            <span data-t='user-question-bt-cancel'>
                                <Button view='pseudo' size='l' onClick={this.closeKvKoModal}>
                                    {i18n('_AUTH_.common.cancel')}
                                </Button>
                            </span>
                        </div>
                    )}
                    <div className='p-control-saveblock-cell-left p-control-saveblock-button'>
                        <ReturnToService retpath={retpath} isModal={this.props.modal} isMobile={isMobile} />
                    </div>
                    <div className='p-control-saveblock-cell-right p-control-saveblock-button'>
                        <span data-t='user-question-bt-save'>
                            <Button type='submit' size='l' view='action' width={isMobile ? 'max' : 'auto'}>
                                {i18n('_AUTH_.common.save')}
                            </Button>
                        </span>
                        <span className='personal-spinner'>
                            <Spin size='s' progress={loading} />
                        </span>
                    </div>
                </div>
            </form>
        );
    }
}

KvKoModal.propTypes = {
    dispatch: PropTypes.func.isRequired,
    modal: PropTypes.bool.isRequired,
    security: PropTypes.object.isRequired,
    settings: PropTypes.object.isRequired,
    common: PropTypes.shape({
        retpath: PropTypes.string
    }).isRequired
};
