import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {Popup} from '@components/Popup';
import escape from 'lodash/escape';
import Subscription from '@blocks/morda/social_info/subscription/subscription.jsx';
import {LINK_CLASS_NAME} from '@blocks/morda/components/link/link.jsx';

import * as extracted from './profile.js';

export default class Profile extends Component {
    deleteButton = React.createRef();

    constructor(props) {
        super(props);

        this.hideAllSettings = extracted.hideAllSettings.bind(this);
        this.showDeletePopup = extracted.showDeletePopupHandler.bind(this);
        this.hideDeletePopup = extracted.hideDeletePopup.bind(this);
        this.deleteProfile = extracted.deleteProfile.bind(this);
    }

    componentDidMount() {
        extracted.scrollToTop.call(this);
    }

    componentWillUnmount() {
        this.hideDeletePopup();
    }

    render() {
        const {profile, isDeletePopupShown = false} = this.props;
        const isAuthAllowed = profile.allowAuth !== 'no_auth';
        const popupInnerHtml = `<a href='${profile.profileLink}' class='${LINK_CLASS_NAME}'>
                                    <div class='social-icon social-icon_${profile.provider.code} \
                                    social-icon_size_s'></div>
                                    ${escape(profile.username)}
                                </a>`;
        const popupHtml =
            profile.provider.code === 'apl'
                ? i18n('ProfileSocial.profile.social.popup-text-apl')
                : i18n('ProfileSocial.profile.social.popup-text', popupInnerHtml);

        return (
            <div className='s-profile'>
                <div className='s-profile__head'>
                    <div className={`social-icon social-icon_${profile.provider.code} s-profile__icon`} />
                    <div className='s-profile__name'>{profile.username}</div>
                </div>
                {Boolean(isAuthAllowed) && Boolean(profile.subscriptions) && Boolean(profile.subscriptions.length) && (
                    <div>
                        <div className='s-profile__privacy-settings-title'>
                            {i18n('Profile.social.privacy-settings')}
                        </div>
                        <div className='s-profile__privacy-settings-description'>
                            {i18n('ProfileSocial.profile.social.description')}
                        </div>
                        {profile.subscriptions.map((subscription, index) => (
                            <Subscription key={index} subscription={subscription} dispatch={this.props.dispatch} />
                        ))}
                    </div>
                )}
                <div className='s-profile__section s-profile__section_delete'>
                    <div className='s-profile__button s-profile__button_left' ref={this.deleteButton}>
                        <Button view='pseudo' size='l' onClick={this.showDeletePopup}>
                            {i18n('ProfileSocial.profile.social.unlink-profile')}
                        </Button>
                    </div>
                    <Popup
                        zIndex={99999}
                        hasTail={true}
                        target='anchor'
                        anchor={this.deleteButton}
                        visible={isDeletePopupShown}
                        directions={['right-center', 'top-left', 'bottom-left']}
                    >
                        <div className='s-profile__delete-popup'>
                            <div
                                className='s-profile__delete-popup-text'
                                dangerouslySetInnerHTML={{__html: popupHtml}}
                            />
                            {profile.provider.code !== 'apl' && (
                                <div className='s-profile__delete-popup-delete-button'>
                                    <Button size='l' view='action' onClick={this.deleteProfile}>
                                        {i18n('Profile.common.unlink')}
                                    </Button>
                                </div>
                            )}
                            <div className='s-profile__delete-popup-cancel-button'>
                                <Button
                                    view={profile.provider.code === 'apl' ? 'action' : 'pseudo'}
                                    size='l'
                                    onClick={this.hideDeletePopup}
                                >
                                    {profile.provider.code === 'apl'
                                        ? i18n('Frontend.close')
                                        : i18n('_AUTH_.common.cancel')}
                                </Button>
                            </div>
                        </div>
                    </Popup>
                    <div className='s-profile__delete-description'>
                        {i18n('Profile.social.unlink-profile-description')}
                    </div>
                </div>
                <div className='s-profile__section'>
                    {Boolean(this.props.modal) && (
                        <Button view='pseudo' size='l' onClick={this.hideAllSettings}>
                            {i18n('_AUTH_.common.back')}
                        </Button>
                    )}
                </div>
            </div>
        );
    }
}

Profile.propTypes = {
    dispatch: PropTypes.func.isRequired,
    profile: PropTypes.object.isRequired,
    modal: PropTypes.bool.isRequired,
    isDeletePopupShown: PropTypes.bool
};
