import assign from 'lodash/assign';
import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {bindActionCreators} from 'redux';
import {Account} from '../account/account.jsx';
import filter from 'lodash/filter';
import classnames from 'classnames';
import {
    domikIsLoading,
    changeActiveAccount,
    forgetAccount,
    loginSuggestedAccount,
    logoutAccount,
    redirectToRetpath,
    toggleAccountListEditing,
    ONE_DOMIK_GOAL_PREFIX
} from '../actions';
import {SwitchModeToAddLink} from '../switch/switch.jsx';
import metrics from '../../metrics';

class AccountList extends Component {
    static mapStateToProps(state) {
        const oneDomik = state.one_domik;
        const settings = state.settings;
        const common = state.common;

        const retpath = oneDomik.formIsUpgraded ? common.loginStatusRetpath : common.retpath || common.currentUrl;

        return {
            accounts: oneDomik.unitedAccounts,
            avatar: settings.avatar,
            defaultUid: oneDomik.authorizedAccountsDefaultUid,
            isAccountListEditing: oneDomik.isAccountListEditing,
            embeddedAuthUrl: common.embeddedauth_url,
            addUserUrl: common.addUserUrl,
            yu: common.yu,
            retpath
        };
    }

    static mapDispatchToProps(dispatch) {
        const actionCreators = {
            domikIsLoading,
            changeActiveAccount,
            forgetAccount,
            loginSuggestedAccount,
            logoutAccount,
            redirectToRetpath,
            toggleAccountListEditing
        };

        return assign(bindActionCreators(actionCreators, dispatch), {dispatch});
    }

    constructor(props) {
        super(props);

        this.switchToSuggestedAccount = this.switchToSuggestedAccount.bind(this);
        this.switchToAddingAccount = this.switchToAddingAccount.bind(this);
        this.formatUrl = this.formatUrl.bind(this);
    }

    switchToSuggestedAccount(account) {
        metrics.goal(`${ONE_DOMIK_GOAL_PREFIX}_choose_suggest`);

        this.props.domikIsLoading(true);
        this.props.loginSuggestedAccount(account);
    }

    switchToAddingAccount() {
        metrics.goal(`${ONE_DOMIK_GOAL_PREFIX}_choose_other`);
        this.props.switchLinkOnclick();
    }

    formatUrl(uid, action) {
        const basicUrl = this.props.embeddedAuthUrl;
        const yu = this.props.yu;
        const retpath = this.props.retpath;

        if (['logout', 'change_default'].indexOf(action) !== -1) {
            return `${basicUrl}&yu=${yu}&action=${action}&uid=${uid}&retpath=${retpath}`;
        }

        return '';
    }

    render() {
        const props = this.props;
        const accounts = props.accounts;
        const defaultUid = props.defaultUid;
        // const switchLinkOnclick = props.switchLinkOnclick;
        const isTouch = props.isTouch;
        const avatar = props.avatar;
        const switchToSuggestedAccount = this.switchToSuggestedAccount;
        const switchToAddingAccount = this.switchToAddingAccount;
        const formatUrl = this.formatUrl;

        const authorizedAccounts = filter(accounts, {status: 'VALID'});
        const defaultAccount = filter(authorizedAccounts, (o) => o.uid === defaultUid)[0];
        const unAuthorizedAccounts = filter(accounts, (o) => o.status !== 'VALID');
        const suggestedAccounts = filter(unAuthorizedAccounts, (o) => typeof o.allowed_auth_methods !== 'undefined');
        const invalidAccounts = filter(unAuthorizedAccounts, (o) => typeof o.allowed_auth_methods === 'undefined');

        const accountListClassNames = {
            'passport-AccountList': true,
            'passport-Domik-Content': true,
            'passport-Domik-Content_wide': true,
            'passport-AccountList_editting': props.isAccountListEditing
        };

        return (
            <div className='passport-AccountList-h'>
                <h2 className='passport-Domik-Title'>
                    {authorizedAccounts.length === 0
                        ? i18n('_AUTH_.AccountList.title.auth')
                        : i18n('OneDomik.AccountList.title.adduser')}
                </h2>

                <ul className={classnames(accountListClassNames)}>
                    {Boolean(defaultAccount) && (
                        <Account
                            avatar={avatar}
                            isAuthorized={true}
                            isDefault={true}
                            isAccountListEditing={props.isAccountListEditing}
                            isTouch={isTouch}
                            key={defaultAccount.uid}
                            logoutUrl={formatUrl(defaultAccount.uid, 'logout')}
                            onClick={() => props.redirectToRetpath()}
                            onCrossClick={() => props.logoutAccount(defaultAccount.uid)}
                            {...defaultAccount}
                        />
                    )}

                    {authorizedAccounts.map(function(account) {
                        const isAuthorized = true;
                        const isDefault = defaultUid === account.uid;

                        if (!isDefault) {
                            return (
                                <Account
                                    avatar={avatar}
                                    isAuthorized={isAuthorized}
                                    isAccountListEditing={props.isAccountListEditing}
                                    isTouch={isTouch}
                                    key={account.uid}
                                    logoutUrl={formatUrl(account.uid, 'logout')}
                                    changeDefaultUrl={formatUrl(account.uid, 'change_default')}
                                    onClick={() => {
                                        props.domikIsLoading(true);
                                        props.changeActiveAccount(account.uid);
                                    }}
                                    onCrossClick={() => props.logoutAccount(account.uid)}
                                    {...account}
                                />
                            );
                        }
                        return false;
                    })}

                    {suggestedAccounts.map(function(account) {
                        const isAuthorized = false;

                        return (
                            <Account
                                avatar={avatar}
                                isAuthorized={isAuthorized}
                                isAccountListEditing={props.isAccountListEditing}
                                isDefault={false}
                                isTouch={isTouch}
                                key={account.uid}
                                onClick={() => switchToSuggestedAccount(account)}
                                onCrossClick={() => props.forgetAccount(account.uid)}
                                {...account}
                            />
                        );
                    })}

                    {invalidAccounts.map(function(account) {
                        const isAuthorized = false;

                        return (
                            <Account
                                avatar={avatar}
                                isAuthorized={isAuthorized}
                                isAccountListEditing={props.isAccountListEditing}
                                isDefault={false}
                                isTouch={isTouch}
                                key={account.uid}
                                onClick={() => switchToSuggestedAccount(account)}
                                onCrossClick={() => props.logoutAccount(account.uid)}
                                {...account}
                            />
                        );
                    })}

                    {Boolean(authorizedAccounts.length < 15) && (
                        <SwitchModeToAddLink
                            linkText={i18n('OneDomik.AccountList.addAccount')}
                            addUserUrl={this.props.addUserUrl}
                            onMouseDown={this.props.onMouseDown}
                            onClick={switchToAddingAccount}
                        />
                    )}
                </ul>

                {Boolean(isTouch) && (
                    <EditListButton
                        enableAccountListEditing={() => props.toggleAccountListEditing(true)}
                        disableAccountListEditing={() => props.toggleAccountListEditing(false)}
                        isAccountListEditing={props.isAccountListEditing}
                    />
                )}
            </div>
        );
    }
}

class EditListButton extends Component {
    render() {
        const props = this.props;

        const buttonClassNames = {
            'passport-Button': true,
            'passport-Button_edit': true,
            'passport-Button_active': props.isAccountListEditing,
            'passport-AccountList-EditButton': true
        };

        return (
            <button type='button' className={classnames(buttonClassNames)}>
                <span className='passport-Button-Content'>
                    <span
                        onClick={props.enableAccountListEditing}
                        className='passport-Button-Text passport-Button-Text_edit'
                    >
                        {i18n('OneDomik.AccountList.ButtonText.edit')}
                    </span>
                    <span
                        onClick={props.disableAccountListEditing}
                        className='passport-Button-Text passport-Button-Text_done'
                    >
                        {i18n('OneDomik.AccountList.ButtonText.done')}
                    </span>
                </span>
            </button>
        );
    }
}

AccountList.propTypes = {
    accounts: PropTypes.object,
    avatar: PropTypes.object,
    defaultUid: PropTypes.string,
    retpath: PropTypes.string,
    yu: PropTypes.string,
    switchLinkOnclick: PropTypes.func,
    loginSuggestedAccount: PropTypes.func,
    onMouseDown: PropTypes.func,
    disableAccountListEditing: PropTypes.func,
    enableAccountListEditing: PropTypes.func,
    toggleAccountListEditing: PropTypes.func,
    logoutAccount: PropTypes.func,
    forgetAccount: PropTypes.func,
    changeActiveAccount: PropTypes.func,
    redirectToRetpath: PropTypes.func,
    isAccountListEditing: PropTypes.bool,
    domikIsLoading: PropTypes.bool,
    isTouch: PropTypes.bool,
    addUserUrl: PropTypes.string,
    embeddedAuthUrl: PropTypes.string
};

EditListButton.propTypes = {
    isAccountListEditing: PropTypes.bool,
    disableAccountListEditing: PropTypes.func,
    enableAccountListEditing: PropTypes.func
};

export const ConnectedAccountList = connect(AccountList.mapStateToProps, AccountList.mapDispatchToProps)(AccountList);
