import React, {Component} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {connect} from 'react-redux';
import {Input} from '@components/Input';
import {Icon} from '@components/IconLego';
import {Button} from '@components/Button';
import {Popup} from '@components/Popup';
import {Spin} from '@components/Spin';
import {Captcha} from '@components/Captcha';
import RevokersModal from './revokers_modal.jsx';
import PasswordLine from './password_line.jsx';

import {
    changeText,
    submitPassword,
    showTipsPopup,
    toggleShowPassword,
    getPasswordStrength,
    validateField,
    setError
} from './actions';

const errorMessages = {
    currentPassword: {
        empty: i18n('_AUTH_.errors.required'),
        not_matched: i18n('Frontend.password_current_errors_invalid'),
        invalid: i18n('_AUTH_.phones.errors.phone.invalid'),
        found_in_history: i18n('_AUTH_.password_errors_foundinhistory'),
        equals_previous: i18n('_AUTH_.password_errors_equals_previous')
    },
    newPassword: {
        empty: i18n('_AUTH_.errors.required')
    },
    repeatPassword: {
        empty: i18n('_AUTH_.errors.required'),
        not_matched: i18n('_AUTH_.password_confirm_errors_notequal'),
        matched: i18n('Frontend.password_confirm_ok')
    },
    captcha: {
        empty: i18n('_AUTH_.errors.required'),
        not_matched: i18n('Profile.domik.captcha-not-matched')
    }
};

const messages = {
    tooshort: i18n('_AUTH_.password_errors_tooshort'),
    toolong: i18n('_AUTH_.password_errors_toolong'),
    tooweak: i18n('_AUTH_.password_errors_tooweak'),
    weak: i18n('_AUTH_.password_errors_weak'),
    prohibitedsymbols: i18n('_AUTH_.password_errors_prohibitedsymbols'),
    likelogin: i18n('_AUTH_.password_errors_likelogin'),
    likephonenumber: i18n('_AUTH_.password_errors_likephonenumber'),
    likeOldPassword: i18n('_AUTH_.password_errors_equals_previous'),
    foundInHistory: i18n('_AUTH_.password_errors_foundinhistory'),
    strong: i18n('Frontend.password_ok')
};

const tipMessages = {
    tooshort: i18n('Profile.password.tips.more-than-six'),
    tooweak: i18n('Profile.password.tips.too-weak'),
    forbidden_words: i18n('Profile.password.tips.forbidden-words'),
    weak: i18n('Profile.password.tips.weak'),
    strong: i18n('Profile.password.tips.strong'),
    super_strong: i18n('Profile.password.tips.super-strong'),
    prohibitedsymbols: i18n('Profile.password.tips.prohibitedsymbols')
};

class Password extends Component {
    newPasswordRef = React.createRef();

    static mapStateToProps(state) {
        const {password, captcha, settings, common, password_status} = state;
        const {ua, language, staticPath} = settings;
        const {retpath = '/profile'} = common;
        const yandexuid = state.common.yandexuid;
        const isMobile = Boolean((ua.isMobile || ua.isTouch) && !ua.isTablet);
        const isMobileCaptcha = Boolean(ua.isMobile || ua.isTablet);

        return {
            retpath,
            yandexuid,
            staticPath,
            password,
            password_status,
            captcha,
            isMobile,
            isMobileCaptcha,
            language
        };
    }

    constructor(props) {
        super(props);

        this.onFormSubmit = this.onFormSubmit.bind(this);
        this.onFieldBlur = this.onFieldBlur.bind(this);
        this.onFieldFocus = this.onFieldFocus.bind(this);
        this.onNewPasswordFocus = this.onNewPasswordFocus.bind(this);
        this.onNewPasswordBlur = this.onNewPasswordBlur.bind(this);
        this.onNewPasswordChange = this.onNewPasswordChange.bind(this);
        this.onEyeClick = this.onEyeClick.bind(this);
        this.onTextChange = this.onTextChange.bind(this);

        this.eye = `${props.staticPath}i/icon_eye.svg`;
        this.eyeClosed = `${props.staticPath}i/icon_eye_closed.svg`;
        this.checkGreen = `${props.staticPath}i/check_green.svg`;
        this.checkGrey = `${props.staticPath}i/check_grey.svg`;
    }

    onFormSubmit(event) {
        event.preventDefault();
        this.props.dispatch(submitPassword());
    }

    onNewPasswordFocus(event) {
        const {name} = event.target;

        this.props.dispatch(showTipsPopup(true));
        this.props.dispatch(setError(null, name));
    }

    onNewPasswordBlur(event) {
        const {value, name} = event.target;

        this.props.dispatch(showTipsPopup(false));
        this.props.dispatch(validateField(value, name));
    }

    onNewPasswordChange(event) {
        const {target = {}} = event;
        const {value, name} = target;

        this.props.dispatch(changeText({text: value, name}));
        this.props.dispatch(getPasswordStrength(value));
    }

    onFieldBlur(event) {
        const {value, name} = event.target;

        this.props.dispatch(validateField(value, name));
    }

    onFieldFocus(event) {
        const {name} = event.target;

        this.props.dispatch(setError(null, name));
    }

    onEyeClick() {
        this.props.dispatch(toggleShowPassword());
    }

    onTextChange(event) {
        const {target = {}} = event;
        const {value, name} = target;

        this.props.dispatch(changeText({text: value, name}));
    }

    componentDidMount() {
        const icon = this.newPasswordRef.querySelector('.icon');

        icon.onclick = this.onEyeClick.bind(this);
    }

    // renderTip(tip) {
    //     const fullfilled = this.props.password.fullFilledTips.includes(tip);
    //     const check = fullfilled ? this.checkGreen : this.checkGrey;
    //
    //     return (
    //         <li className={classnames(
    //             'changepass__tips-list-item',
    //             {
    //                 'changepass__tips-list-item_fullfilled': fullfilled
    //             }
    //         )}>
    //             <img className='changepass__tips-list-item-check' src={check}/>
    //             <div>{tips[tip]}</div>
    //         </li>
    //     );
    // }

    checkForNumbersOrSpecialSymbols(text) {
        // eslint-disable-next-line no-useless-escape
        return /[0-9.,`!@#$%^&*()":;?_+=\\|\/{}\[\]<>]/.test(text);
    }

    checkForNumbersAndCases(text) {
        return [/\d/, /[a-z]/, /[A-Z]/].every(function(p) {
            return p.test(text);
        });
    }

    renderTips() {
        const {
            newPassword,
            passwordStrength: {code, value}
        } = this.props.password;

        if (!(newPassword.length > 0)) {
            return null;
        }

        return (
            <div className='changepass__tips'>
                <h3 className='changepass__tips-title'>
                    {code === 'weak' && value === 0 ? messages.tooweak : messages[code]}
                </h3>
                <PasswordLine text={newPassword} strength={value} code={code} />
                <div className='changepass__tips-text' aria-live='polite'>
                    {(() => {
                        if (value === -1 && this.checkForNumbersOrSpecialSymbols(newPassword)) {
                            return tipMessages.forbidden_words;
                        }

                        if (code === 'weak' && value === 0) {
                            return tipMessages.tooweak;
                        }

                        if (code === 'strong' && this.checkForNumbersAndCases(newPassword)) {
                            return tipMessages.super_strong;
                        }

                        return tipMessages[code];
                    })()}
                </div>
            </div>
        );
    }

    renderError(fieldName) {
        const error = this.props.password.errors[fieldName];

        if (!error) {
            return null;
        }

        return (
            <div
                className={classnames('p-control-error-block', {
                    'p-control-success-block': error === 'matched'
                })}
            >
                {errorMessages[fieldName][error]}
            </div>
        );
    }

    renderForm() {
        const {
            captcha,
            retpath,
            password,
            dispatch,
            language,
            isMobile,
            yandexuid,
            password_status,
            isMobileCaptcha
        } = this.props;
        const {
            appPasswords,
            devicesList,
            allOtherTokens,
            showRevokersModal,
            openedSections,
            showPassword,
            loading,
            currentPassword,
            newPassword,
            repeatPassword,
            showPasswordTips,
            deletingTokens,
            showWebSessions
        } = password;
        const eye = <Icon url={showPassword ? this.eyeClosed : this.eye} />;
        const allowSelect = password_status.revokers.allow_select;

        return (
            <div>
                <h1 className='changepass__title'>{i18n('Frontend.title.changepassword')}</h1>
                <div className='changepass__current-password'>
                    <label className='changepass__label' htmlFor='currentPassword'>
                        {i18n('_AUTH_.field_password_current')}:
                    </label>
                    <Input
                        size='l'
                        type='password'
                        value={currentPassword}
                        onChange={this.onTextChange}
                        onBlur={this.onFieldBlur}
                        onFocus={this.onFieldFocus}
                        id='currentPassword'
                        name='currentPassword'
                        autoFocus={true}
                    />
                    {this.renderError('currentPassword')}
                </div>
                <div className='changepass__new-password-block'>
                    <div className='changepass__new-password'>
                        <label className='changepass__label' htmlFor='newPassword'>
                            {i18n('Frontend.field_password_new')}:
                        </label>
                        <Input
                            size='l'
                            controlRef={this.newPasswordRef}
                            type={showPassword ? 'text' : 'password'}
                            value={newPassword}
                            iconRight={eye}
                            onFocus={this.onNewPasswordFocus}
                            onBlur={this.onNewPasswordBlur}
                            onChange={this.onNewPasswordChange}
                            id='newPassword'
                            name='newPassword'
                        />
                        {this.renderError('newPassword')}
                        {isMobile ? (
                            this.renderTips()
                        ) : (
                            <Popup
                                target='anchor'
                                visible={showPasswordTips}
                                anchor={this.newPasswordRef}
                                directions={['right-center']}
                            >
                                {this.renderTips()}
                            </Popup>
                        )}
                    </div>
                    <div className='changepass__repeat-password'>
                        <label className='changepass__label' htmlFor='repeatPassword'>
                            {i18n('_AUTH_.field_password_confrimnew')}:
                        </label>
                        <Input
                            size='l'
                            type={showPassword ? 'text' : 'password'}
                            value={repeatPassword}
                            onChange={this.onTextChange}
                            id='repeatPassword'
                            name='repeatPassword'
                        />
                        {this.renderError('repeatPassword')}
                    </div>
                </div>
                <div className='changepass__captcha-block'>
                    <div className='changepass__captcha-input'>
                        <label className='changepass__label' htmlFor='captcha'>
                            {i18n('_AUTH_.Errors.captcha.required')}:
                        </label>
                        <Input
                            size='l'
                            value={password.captcha}
                            onChange={this.onTextChange}
                            onFocus={this.onFieldFocus}
                            onBlur={this.onFieldBlur}
                            id='captcha'
                            name='captcha'
                        />
                        {this.renderError('captcha')}
                    </div>
                    <div className='changepass__captcha'>
                        <Captcha
                            isMobile={isMobileCaptcha}
                            lang={language}
                            introSound={captcha.introSound}
                            captchaSound={captcha.captchaSound}
                            imageUrl={captcha.imageUrl}
                            loading={captcha.loading}
                            loadingAudio={captcha.loadingAudio}
                            playing={captcha.playing}
                            type={captcha.type}
                            dispatch={dispatch}
                            visible={true}
                        />
                    </div>
                </div>
                <div className='changepass__controls'>
                    {loading === 'form' ? (
                        <Spin size='m' progress={true} />
                    ) : (
                        <Button view='action' type='submit' size='l' width={isMobile ? 'max' : 'auto'}>
                            {i18n('Profile.common.continue')}
                        </Button>
                    )}
                </div>
                <RevokersModal
                    isMobile={false}
                    retpath={retpath}
                    yu={yandexuid}
                    dispatch={dispatch}
                    showWebSessions={showWebSessions}
                    deletingTokens={deletingTokens}
                    openedSections={openedSections}
                    appPasswords={appPasswords}
                    devicesList={devicesList}
                    allOtherTokens={allOtherTokens}
                    allowSelect={allowSelect}
                    visible={showRevokersModal}
                />
            </div>
        );
    }

    render() {
        const {password, retpath, dispatch, isMobile, yandexuid, password_status} = this.props;
        const {
            appPasswords,
            devicesList,
            allOtherTokens,
            showRevokersModal,
            openedSections,
            deletingTokens,
            showWebSessions
        } = password;
        const allowSelect = password_status.revokers.allow_select;

        return (
            <form className='changepass' onSubmit={this.onFormSubmit}>
                {isMobile && showRevokersModal ? (
                    <RevokersModal
                        retpath={retpath}
                        isMobile={isMobile}
                        yu={yandexuid}
                        dispatch={dispatch}
                        showWebSessions={showWebSessions}
                        deletingTokens={deletingTokens}
                        openedSections={openedSections}
                        appPasswords={appPasswords}
                        devicesList={devicesList}
                        allOtherTokens={allOtherTokens}
                        allowSelect={allowSelect}
                        visible={showRevokersModal}
                    />
                ) : (
                    this.renderForm()
                )}
            </form>
        );
    }
}

Password.propTypes = {
    staticPath: PropTypes.string,
    captcha: PropTypes.object,
    dispatch: PropTypes.func.isRequired,
    isMobileCaptcha: PropTypes.bool,
    isMobile: PropTypes.bool,
    language: PropTypes.string,
    password: PropTypes.string,
    password_status: PropTypes.string,
    retpath: PropTypes.string,
    yandexuid: PropTypes.string
};

const connectedPassword = connect(Password.mapStateToProps)(Password);

export {connectedPassword as Password};
