import React, {Component} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import {Modal} from '@components/Modal';
import {Button} from '@components/Button';
import {toggleSection, showModal, revokeTokens, revokeGlobal} from './actions';

const title = {
    devicesList: i18n('Profile.devices.list'),
    allOtherTokens: i18n('Profile.devices.other'),
    appPasswords: i18n('Profile.access.appwd-title')
};

class RevokersModal extends Component {
    constructor(props) {
        super(props);

        this.onSectionClick = this.onSectionClick.bind(this);
        this.onSectionExitClick = this.onSectionExitClick.bind(this);
        this.onTokenExitClick = this.onTokenExitClick.bind(this);
        this.onGlobalExit = this.onGlobalExit.bind(this);
        this.onBrowserExit = this.onBrowserExit.bind(this);
        this.onCloseClick = this.onCloseClick.bind(this);
    }

    onSectionClick(event) {
        const section = event.target.dataset.section;

        this.props.dispatch(toggleSection(section));
    }

    onSectionExitClick(event) {
        const section = event.currentTarget.id;
        const tokens = [];

        if (section) {
            this.props[section].forEach(function(elem) {
                switch (section) {
                    case 'devicesList': {
                        elem.tokens.forEach(function(token) {
                            tokens.push(token.id);
                        });
                        break;
                    }
                    case 'allOtherTokens':
                    case 'appPasswords': {
                        tokens.push(elem.id);
                        break;
                    }
                    default:
                        throw Error('Unexpected section');
                }
            });
        }

        this.props.dispatch(revokeTokens(tokens, section));
    }

    onTokenExitClick(event) {
        const [section, id] = event.currentTarget.id.split('.');
        const tokens = [];

        if (section === 'devicesList') {
            this.props.devicesList.forEach(function(device) {
                if (id === device.deviceName) {
                    device.tokens.forEach(function(token) {
                        tokens.push(token.id);
                    });
                }
            });
        } else {
            tokens.push(parseInt(id, 10));
        }

        this.props.dispatch(revokeTokens(tokens, section));
    }

    onGlobalExit() {
        this.props.dispatch(revokeGlobal('all'));
    }

    onBrowserExit() {
        this.props.dispatch(revokeGlobal('webSessions'));
    }

    onCloseClick() {
        this.props.dispatch(showModal(false));
    }

    renderSectionTitle(section) {
        const deletingTokens = this.props.deletingTokens;

        return (
            <div
                className={classnames('changepass__modal-section', {
                    'changepass__modal-section_opened': this.props.openedSections[section]
                })}
            >
                <div
                    className={classnames('changepass__modal-arrow', {
                        'changepass__modal-arrow_opened': this.props.openedSections[section]
                    })}
                    data-section={section}
                    onClick={this.onSectionClick}
                />
                <h3 className='changepass__modal-section-title'>{title[section]}</h3>
                <Button size='l' view='pseudo' id={section} onClick={this.onSectionExitClick} disabled={deletingTokens}>
                    {i18n('Profile.logout.all')}
                </Button>
            </div>
        );
    }

    render() {
        const {visible, isMobile} = this.props;

        return (
            <div>
                <Modal visible={visible && !isMobile} onCloserClick={this.onCloseClick}>
                    {Boolean(visible && !isMobile) && this.renderRevokers()}
                </Modal>
                {Boolean(isMobile && visible) && this.renderRevokers()}
            </div>
        );
    }

    renderRevokers() {
        const {
            devicesList,
            showWebSessions,
            allOtherTokens,
            deletingTokens,
            retpath,
            appPasswords,
            openedSections,
            allowSelect
        } = this.props;

        return (
            <section className='changepass__modal'>
                <h1 className='changepass__modal-title'>{i18n('Restore.restore.success')}</h1>
                <h2 className='changepass__modal-subtitle'>{i18n('Profile.logout.choose')}</h2>
                {Boolean(devicesList.length > 0 && allowSelect) && this.renderSectionTitle('devicesList')}
                {Boolean(openedSections.devicesList && allowSelect) &&
                    devicesList.map(
                        (device, deviceIndex) =>
                            Boolean(!device.duble) && (
                                <div key={deviceIndex} className='changepass__modal-section-item'>
                                    <div className='changepass__modal-section-item-text'>{device.deviceName}</div>
                                    <div
                                        className={classnames(
                                            'changepass__modal-section-item-icons',
                                            'changepass__modal-section-item-icons_devices'
                                        )}
                                    >
                                        {device.tokens.map(
                                            (token, index) =>
                                                Boolean(token.client.iconUrl) && (
                                                    <div
                                                        key={index}
                                                        className={classnames(
                                                            'changepass__modal-section-item-icon',
                                                            'changepass__modal-section-item-icon_devices'
                                                        )}
                                                        style={{backgroundImage: `url('${token.client.iconUrl}')`}}
                                                    />
                                                )
                                        )}
                                    </div>
                                    <Button
                                        size='l'
                                        view='pseudo'
                                        id={`devicesList.${device.deviceName}`}
                                        onClick={this.onTokenExitClick}
                                        disabled={deletingTokens}
                                    >
                                        {i18n('Profile.logout.exit')}
                                    </Button>
                                </div>
                            )
                    )}

                {Boolean(allOtherTokens.length > 0 && allowSelect) && this.renderSectionTitle('allOtherTokens')}

                {Boolean(openedSections.allOtherTokens && allowSelect) &&
                    allOtherTokens.map((token, index) => (
                        <div key={index} className='changepass__modal-section-item'>
                            <div className='changepass__modal-section-item-text'>{token.client.title}</div>
                            <div className='changepass__modal-section-item-icons'>
                                {Boolean(token.client.iconUrl && token.client.iconUrl) && (
                                    <div
                                        className='changepass__modal-section-item-icon'
                                        style={{backgroundImage: `url('${token.client.iconUrl}')`}}
                                    />
                                )}
                            </div>
                            <Button
                                size='l'
                                view='pseudo'
                                id={`allOtherTokens.${token.id}`}
                                onClick={this.onTokenExitClick}
                                disabled={deletingTokens}
                            >
                                {i18n('Profile.logout.exit')}
                            </Button>
                        </div>
                    ))}

                {Boolean(appPasswords.length > 0 && allowSelect) && this.renderSectionTitle('appPasswords')}

                {Boolean(openedSections.appPasswords && allowSelect) &&
                    appPasswords.map((password, index) => (
                        <div key={index} className='changepass__modal-section-item'>
                            <div
                                className={classnames(
                                    'changepass__modal-section-item-text',
                                    'changepass__modal-section-item-text_app-passwords'
                                )}
                            >
                                {password.device_name}
                            </div>
                            <Button
                                size='l'
                                view='pseudo'
                                id={`appPasswords.${password.id}`}
                                onClick={this.onTokenExitClick}
                                disabled={deletingTokens}
                            >
                                {i18n('Profile.logout.exit')}
                            </Button>
                        </div>
                    ))}

                {Boolean(showWebSessions) && (
                    <div className='changepass__modal-section changepass__modal-section_browsers'>
                        <h3 className='changepass__modal-section-title'>{i18n('Profile.logout.browsers')}</h3>
                        <Button view='pseudo' size='l' onClick={this.onBrowserExit} disabled={deletingTokens}>
                            {i18n('Profile.logout.all')}
                        </Button>
                    </div>
                )}

                <div className='changepass__modal-controls'>
                    <Button view='pseudo' size='l' onClick={this.onGlobalExit} disabled={deletingTokens}>
                        {i18n('Profile.logout.everywhere')}
                    </Button>
                    <Button view='action' size='l' type='link' url={retpath}>
                        {i18n('OneDomik.AccountList.ButtonText.done')}
                    </Button>
                </div>
            </section>
        );
    }
}

RevokersModal.propTypes = {
    dispatch: PropTypes.func.isRequired,
    deletingTokens: PropTypes.bool,
    openedSections: PropTypes.shape({
        devicesList: PropTypes.array,
        appPasswords: PropTypes.array,
        allOtherTokens: PropTypes.array
    }),
    visible: PropTypes.bool,
    isMobile: PropTypes.bool,
    showWebSessions: PropTypes.bool,
    allOtherTokens: PropTypes.array,
    appPasswords: PropTypes.array,
    devicesList: PropTypes.array,
    retpath: PropTypes.string,
    allowSelect: PropTypes.bool
};

export default RevokersModal;
