import api from '@blocks/api';
import {getFormValues} from '@blocks/selectors';
import {PAGES, SECONDS_TO_NEXT_CODE_SEND, OPERATION_TYPES, ERRORS_MAP, MODAL_CONTENT_RENDER_TIMEOUT} from '../const';
import {getUnixTimeStamp, getError, getCookie} from '@blocks/utils';
import {FIELDS_NAMES} from '@components/Field/names';
import {updateErrors, updateValues, togglePasswordVisibilityState} from '@blocks/actions/form';
import {updateTrack} from '@blocks/common/actions';
import reloadCaptcha from '@components/Captcha/actions/reloadCaptcha';
import {toggleEnterWithoutPassword} from '@blocks/morda/personal_info/effects/enter_without_password';

// TODO распилить файл, отделить тематические запросы в отдельные апи

export const SET_PHONES_PAGE = 'SET_PHONES_PAGE';
export const SET_PHONES_ERROR = 'SET_PHONES_ERROR';
export const SET_PHONES_LOADING = 'SET_PHONES_LOADING';
export const SMS_2FA_ENABLED = 'SMS_2FA_ENABLED';
export const SET_PHONE_ACCESS = 'SET_PHONE_ACCESS';
export const SET_OPERATION = 'SET_OPERATION';
export const SET_DENY_RESEND_UNTIL = 'SET_DENY_RESEND_UNTIL';
export const SET_PASSWORD_REQUIRED = 'SET_PASSWORD_REQUIRED';
export const SET_CODE_REQUIRED = 'SET_CODE_REQUIRED';
export const SET_PHONES_CAPTCHA_REQUIRED = 'SET_PHONES_CAPTCHA_REQUIRED';
export const SET_CURRENT_PHONE_ID = 'SET_CURRENT_PHONE_ID';
export const SET_CONFIRMATION_CODE_SENT_STATUS = 'SET_CONFIRMATION_CODE_SENT_STATUS';
export const SET_PROCESSED_NUMBER = 'SET_PROCESSED_NUMBER';
export const UPDATE_PHONES = 'UPDATE_PHONES';
export const RESET_PHONES_TEMP_VALUES = 'RESET_PHONES_TEMP_VALUES';
export const SET_REPLACE_SECOND_STEP = 'SET_REPLACE_SECOND_STEP';
export const SET_REPLACE_ADD_PHONE_STEP = 'SET_REPLACE_ADD_PHONE_STEP';
export const SET_IS_LOGIN_ALIAS_STATE = 'SET_IS_LOGIN_ALIAS_STATE';
export const SET_IS_EMAIL_ALIAS_STATE = 'SET_IS_EMAIL_ALIAS_STATE';
export const SET_NOTIFICATIONS_STATE = 'SET_NOTIFICATIONS_STATE';
export const SET_CURRENT_PHONE = 'SET_CURRENT_PHONE';

export const sms2faEnabled = (enabled = true) => ({type: SMS_2FA_ENABLED, enabled});
export const setPhonesPage = (payload) => ({type: SET_PHONES_PAGE, payload});
export const setPhonesError = (payload) => ({type: SET_PHONES_ERROR, payload});
export const setPhonesLoading = (payload) => ({type: SET_PHONES_LOADING, payload});
export const setPhoneAccess = () => ({type: SET_PHONE_ACCESS});
export const setOperation = ({isSecure, phoneId, id, type} = {}) => ({
    type: SET_OPERATION,
    payload: {isSecure, phoneId, id, type}
});
export const setDenyResendUntil = (payload) => ({type: SET_DENY_RESEND_UNTIL, payload});
export const setPasswordRequired = (payload) => ({type: SET_PASSWORD_REQUIRED, payload});
export const setCodeRequired = (payload) => ({type: SET_CODE_REQUIRED, payload});
export const setCaptchaRequired = (payload) => ({type: SET_PHONES_CAPTCHA_REQUIRED, payload});
export const setCurrentPhoneId = ({phoneId, isSecure = true} = {}) => ({
    type: SET_CURRENT_PHONE_ID,
    payload: {phoneId, isSecure}
});
export const setConfirmationCodeSentStatus = (payload) => ({type: SET_CONFIRMATION_CODE_SENT_STATUS, payload});
export const setProcessedNumber = (payload) => ({type: SET_PROCESSED_NUMBER, payload});
export const updatePhones = (payload) => ({type: UPDATE_PHONES, payload});
export const resetPhonesTempValues = () => ({type: RESET_PHONES_TEMP_VALUES});
export const setReplaceSecondStep = (payload) => ({type: SET_REPLACE_SECOND_STEP, payload});
export const setCurrentPhone = (phoneId) => ({type: SET_CURRENT_PHONE, payload: {phoneId}});
export const setReplaceAddPhoneStep = (payload) => ({type: SET_REPLACE_ADD_PHONE_STEP, payload});
export const setIsLoginAliasState = (payload) => ({type: SET_IS_LOGIN_ALIAS_STATE, payload});
export const setIsEmailAliasState = (payload) => ({type: SET_IS_EMAIL_ALIAS_STATE, payload});
export const setNotificationsState = (payload = {}) => ({type: SET_NOTIFICATIONS_STATE, payload});

export const makeRequest = (dispatch, method, data = {}) => {
    dispatch(setPhonesLoading(true));
    return api
        .request(method, data)
        .fail(({errors = []}) => {
            const errorCode = Array.isArray(errors) && errors[0];

            dispatch(setPhonesError(errorCode));

            if (ERRORS_MAP[errorCode]) {
                dispatch(goToPage({page: PAGES.error}));
            }
        })
        .always(() => dispatch(setPhonesLoading(false)));
};

export const removeError = () => (dispatch) => dispatch(setPhonesError(null));

export const onPhonesProcessEnd = ({needToCancelOperation = false} = {}) => (dispatch, getState) => {
    const {phones: {pageInfo: {modal} = {}, error} = {}} = getState();
    const timeout = modal ? MODAL_CONTENT_RENDER_TIMEOUT : 0;

    setTimeout(() => {
        if (needToCancelOperation) {
            dispatch(cancelOperation());
        }
        if (error) {
            dispatch(removeError());
        }
        dispatch(clearFields());
        dispatch(resetPhonesTempValues());
        dispatch(setCaptchaRequired(false));
        dispatch(initTrack());
    }, timeout);
};

const setPage = ({dispatch, page, fullPage, isModal} = {}) => {
    dispatch(
        setPhonesPage(
            isModal
                ? {page: fullPage, modal: page, isModalOpened: Boolean(page)}
                : {page, modal: null, isModalOpened: false}
        )
    );
};

export const goToPage = ({page, canBeModal = true, forceModal = false} = {}) => (dispatch, getState) => {
    const {settings: {isTouch} = {}} = getState();
    const fullPage = PAGES.main;

    setPage({dispatch, page, fullPage, isModal: canBeModal && (!isTouch || forceModal)});
};

export const goToSettingsPage = (phoneId) => (dispatch) => {
    dispatch(setCurrentPhone(phoneId));
    dispatch(goToPage({page: PAGES.phoneSettings}));
};

export const goToMainPage = () => (dispatch) => {
    setPage({dispatch, page: PAGES.main, isModal: false});
};

export const getAccountPhones = () => (dispatch) => {
    return makeRequest(dispatch, 'get-account-phones').done((res = {}) => {
        if (res.status === 'ok') {
            const {account: {phones = {}} = {}} = res;
            const phonesPayload = {
                restore: [],
                other: [],
                otherInReplaceQuarantine: []
            };

            Object.keys(phones).forEach((code) => {
                const phone = phones[code];
                const operation = phone.operation || {};
                const alias = phone.alias || {};
                const {in_quarantine: inQuarantine, id, type, started, finished} = operation;
                const {login_enabled: isLoginAlias, email_enabled: isEmailAlias} = alias;
                const operatonObj = {
                    inQuarantine,
                    id,
                    type,
                    started,
                    finished
                };
                const phoneObj = {
                    id: phone.id,
                    isSecure: Boolean(phone.secured),
                    number: phone.number.masked_international,
                    numberMaskedOriginal: phone.number.masked_original,
                    isDefault: phone.is_default,
                    isAlias: phone.is_alias,
                    needProlong: phone.need_admission,
                    operation: phone.operation ? operatonObj : {},
                    isLoginAlias,
                    isEmailAlias
                };

                if (phone.bound) {
                    if (phone.secured) {
                        return phonesPayload['restore'].push(phoneObj);
                    }

                    if (!phone.secured && phone.operation && phone.operation.type === 'mark') {
                        return phonesPayload['otherInReplaceQuarantine'].push(phoneObj);
                    }

                    phonesPayload['other'].push(phoneObj);
                }
            });

            dispatch(updatePhones(phonesPayload));
        }
    });
};

export const clearFields = () => (dispatch) => {
    [FIELDS_NAMES.PHONE, FIELDS_NAMES.PHONE_CODE, FIELDS_NAMES.PASSWORD].forEach((field) => {
        dispatch(updateValues({field, value: ''}));
        dispatch(updateErrors({field, error: ''}));
    });
};

export const initTrack = () => (dispatch) => {
    makeRequest(dispatch, 'initTrack', {type: 'authorize'}).done((res = {}) => {
        if (res.status === 'ok') {
            const {track_id: trackId} = res;

            dispatch(updateTrack(trackId));
        }
    });
};

export const cancelOperation = (phoneId = '') => (dispatch, getState) => {
    const state = getState();
    const {
        common: {csrf} = {},
        phones: {currentSecurePhoneId, currentSimplePhoneId, other = [], restore = []} = {}
    } = state;

    const curPhoneId = phoneId || currentSecurePhoneId || currentSimplePhoneId;
    const curOperationId = (([...restore, ...other].find((phone) => phone.id === curPhoneId) || {}).operation || {}).id;

    if (!curOperationId) {
        return;
    }

    return makeRequest(dispatch, 'yasms.operation.cancel', {
        csrf_token: csrf,
        id: curOperationId
    }).done(() => {
        dispatch(resetPhonesTempValues());
        dispatch(getAccountPhones());
    });
};

export const phoneBindSecureSubmit = () => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf, track_id} = {}, person: {isPDD} = {}} = state;
    const {phone} = getFormValues(state);
    const mode = isPDD ? 'confirmAndBindSecure' : 'confirmAndBindSecureWithAlias';

    return makeRequest(dispatch, 'phone-confirm-code-submit', {
        number: phone,
        csrf_token: csrf,
        track_id,
        confirm_method: 'by_sms',
        mode,
        enableAliasAsEmail: false
    })
        .done((res = {}) => {
            const {
                deny_resend_until: denyResendUntil,
                number: {masked_international: internationalPhoneNumber} = {},
                is_password_required: isPasswordRequired = false
            } = res;

            dispatch(setPasswordRequired(isPasswordRequired));
            dispatch(setCodeRequired(true));
            dispatch(setDenyResendUntil(denyResendUntil));
            dispatch(setProcessedNumber(internationalPhoneNumber));
            dispatch(setConfirmationCodeSentStatus(true));
        })
        .fail((error = {}) => {
            let errorCode = error.errors && Array.isArray(error.errors) && error.errors[0];

            if (errorCode === 'phone_alias.exist') {
                errorCode = 'phone_secure.bound_and_confirmed';
            }

            dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, errorCode)}));
        });
};

export const toggleAuthBySms = ({notifications} = {}) => (dispatch, getState) => {
    const {person: {loginOptions: {qrCodeLoginForbidden, smsCodeLoginForbidden} = {}} = {}} = getState();

    dispatch(
        toggleEnterWithoutPassword(
            {qrCodeLoginForbidden, smsCodeLoginForbidden: !smsCodeLoginForbidden},
            () => notifications && dispatch(setNotificationsState({isVisible: true, notificationsList: notifications}))
        )
    );
};

export const phoneBindSecureCommit = ({notifications = {}, shouldTurnOnSms2fa = false} = {}) => (
    dispatch,
    getState
) => {
    const state = getState();
    const {
        person: {loginOptions: {smsCodeLoginForbidden} = {}, isPDD} = {},
        common: {csrf, track_id} = {},
        phones: {isPasswordRequired, isCaptchaRequired} = {},
        access: {isSms2faEnabled} = {},
        form: {isPasswordVisible} = {}
    } = state;
    const {phoneCode, password, captcha} = getFormValues(state);
    const mode = isPDD ? 'confirmAndBindSecure' : 'confirmAndBindSecureWithAlias';
    const params = {
        csrf_token: csrf,
        mode,
        password,
        code: phoneCode,
        track_id
    };
    const {notificationWithSms, notificationWithoutSms} = notifications;
    const canToggleAuthBySms = !isPDD && smsCodeLoginForbidden;

    if (captcha && isCaptchaRequired) {
        params.answer = captcha;
    }

    if (isPasswordRequired && !password) {
        dispatch(
            updateErrors({
                field: FIELDS_NAMES.PASSWORD,
                error: getError(FIELDS_NAMES.PASSWORD, 'missingvalue_current')
            })
        );
        return;
    }

    if (isPasswordRequired && /^\s*$/.test(password)) {
        dispatch(
            updateErrors({
                field: FIELDS_NAMES.PASSWORD,
                error: getError(FIELDS_NAMES.PASSWORD, 'password.not_matched')
            })
        );
        return;
    }

    return makeRequest(dispatch, 'phone-confirm-code', params)
        .done(() => {
            if (canToggleAuthBySms) {
                dispatch(toggleAuthBySms({notifications: notificationWithSms}));
            } else {
                notificationWithoutSms &&
                    dispatch(setNotificationsState({isVisible: true, notificationsList: notificationWithoutSms}));
            }

            if (isPasswordVisible) {
                dispatch(togglePasswordVisibilityState());
            }

            if (shouldTurnOnSms2fa) {
                makeRequest(dispatch, 'yasms.sms2faCommit', {
                    csrf_token: csrf,
                    isEnabled: !isSms2faEnabled,
                    track_id
                }).done(() => {
                    dispatch(getAccountPhones());
                    dispatch(onPhonesProcessEnd());
                    dispatch(sms2faEnabled(!isSms2faEnabled));
                });
            } else {
                dispatch(getAccountPhones());
                dispatch(onPhonesProcessEnd());
            }
        })
        .fail((error = {}) => {
            let errorCode = error.errors && Array.isArray(error.errors) && error.errors[0];

            if (errorCode === 'captcha.required' && error.errors[1]) {
                errorCode = error.errors[1];
            }

            const fieldFromError = errorCode && errorCode.split('.')[0];
            const field =
                fieldFromError === 'code' || errorCode === 'confirmations_limit.exceeded'
                    ? FIELDS_NAMES.PHONE_CODE
                    : fieldFromError;

            if (field !== 'captcha') {
                dispatch(updateErrors({field: FIELDS_NAMES.CAPTCHA, error: ''}));
                dispatch(setCaptchaRequired(false));
            }

            if (field === 'captcha' && isCaptchaRequired) {
                dispatch(reloadCaptcha());
            }

            if (errorCode === 'captcha.required' && !isCaptchaRequired) {
                return dispatch(setCaptchaRequired(true));
            }

            dispatch(updateErrors({field, error: getError(field, errorCode)}));
        });
};

export const checkCode = ({isSecure = true} = {}) => (dispatch, getState) => {
    const state = getState();
    const {
        phones: {restore = [], other = [], currentSecurePhoneId, currentSimplePhoneId} = {},
        common: {csrf} = {}
    } = state;
    const arrayForFind = isSecure ? restore : other;
    const phoneIdForFind = isSecure ? currentSecurePhoneId : currentSimplePhoneId;
    const {id} = (arrayForFind.find((phone) => phone.id === phoneIdForFind) || {}).operation || {};
    const {phoneCode} = getFormValues(state);

    return makeRequest(dispatch, 'yasms.code.check', {
        code: phoneCode,
        csrf_token: csrf,
        operationId: id
    }).fail((error = {}) => {
        const errorCode = error.errors && Array.isArray(error.errors) && error.errors[0];

        dispatch(updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: getError(FIELDS_NAMES.PHONE_CODE, errorCode)}));
    });
};

export const checkPassword = ({isSecure = true} = {}) => (dispatch, getState) => {
    const state = getState();
    const {
        phones: {restore = [], other = [], currentSecurePhoneId, currentSimplePhoneId, isCaptchaRequired} = {},
        common: {csrf} = {},
        captcha: {key} = {}
    } = state;
    const arrayForFind = isSecure ? restore : other;
    const phoneIdForFind = isSecure ? currentSecurePhoneId : currentSimplePhoneId;
    const {id} = (arrayForFind.find((phone) => phone.id === phoneIdForFind) || {}).operation || {};
    const {password, captcha} = getFormValues(state);
    const params = {
        password,
        csrf_token: csrf,
        operationId: id
    };

    if (captcha && isCaptchaRequired) {
        params.answer = captcha;
        params.key = key;
    }

    return makeRequest(dispatch, 'yasms.password.check', params).fail((error = {}) => {
        const errorCode = error.errors && Array.isArray(error.errors) && error.errors[0];
        const errorField = ['captcha.required', 'captcha.not_matched', 'captchalocate'].includes(errorCode)
            ? FIELDS_NAMES.CAPTCHA
            : FIELDS_NAMES.PASSWORD;

        if (errorField === 'captcha') {
            dispatch(reloadCaptcha());
        }

        if (errorField !== 'captcha' && isCaptchaRequired) {
            dispatch(setCaptchaRequired(false));
        }

        if (errorCode === 'captcha.required') {
            return dispatch(setCaptchaRequired(true));
        }

        dispatch(updateErrors({field: errorField, error: getError(errorField, errorCode)}));
    });
};

export const resendCode = ({isSecure = true} = {}) => (dispatch, getState) => {
    const state = getState();
    const {
        phones: {restore = [], other = [], currentSecurePhoneId, currentSimplePhoneId} = {},
        common: {csrf} = {}
    } = state;
    const arrayForFind = isSecure ? restore : other;
    const phoneIdForFind = isSecure ? currentSecurePhoneId : currentSimplePhoneId;
    const {id} = (arrayForFind.find((phone) => phone.id === phoneIdForFind) || {}).operation || {};

    return makeRequest(dispatch, 'yasms.code.resend', {
        id,
        csrf_token: csrf
    }).done(() => {
        dispatch(setDenyResendUntil(Number(getUnixTimeStamp()) + SECONDS_TO_NEXT_CODE_SEND));
    });
};

export const removeSecurePhoneSubmit = () => (dispatch, getState) => {
    const state = getState();
    const {phones: {hasPhoneAccess} = {}, common: {csrf} = {}} = state;

    return makeRequest(dispatch, 'yasms.secure.remove.submit', {
        owner: hasPhoneAccess,
        csrf_token: csrf
    }).done((res = {}) => {
        const {operation: {id, phoneId} = {}} = res;

        dispatch(setPasswordRequired(true));
        dispatch(setCodeRequired(hasPhoneAccess ? true : false));
        dispatch(setDenyResendUntil(Number(getUnixTimeStamp()) + SECONDS_TO_NEXT_CODE_SEND));
        dispatch(setOperation({isSecure: true, phoneId, id, type: OPERATION_TYPES.remove}));
        dispatch(setCurrentPhoneId({phoneId, isSecure: true}));
        dispatch(setConfirmationCodeSentStatus(true));
    });
};

export const removeSecurePhoneCommit = () => (dispatch, getState) => {
    const state = getState();
    const {
        phones: {restore = [], currentSecurePhoneId} = {},
        common: {csrf} = {},
        form: {isPasswordVisible} = {}
    } = state;
    const {id} = (restore.find((phone) => phone.id === currentSecurePhoneId) || {}).operation || {};

    return makeRequest(dispatch, 'yasms.secure.remove.commit', {
        id,
        csrf_token: csrf
    }).done(() => {
        dispatch(getAccountPhones());
        dispatch(onPhonesProcessEnd());

        if (isPasswordVisible) {
            dispatch(togglePasswordVisibilityState());
        }
    });
};

export const replaceSecurePhoneSubmit = () => (dispatch, getState) => {
    const state = getState();
    const {phones: {hasPhoneAccess} = {}, common: {csrf} = {}, person: {havePassword} = {}} = state;
    const {phone} = getFormValues(state);

    return makeRequest(dispatch, 'yasms.secure.replace.submit', {
        owner: hasPhoneAccess,
        csrf_token: csrf,
        number: phone
    })
        .done((res = {}) => {
            const {
                secure_phone: {operation: {id: secureOperationId} = {}, id: securePhoneId} = {},
                simple_phone: {operation: {id: simpleOperationId} = {}, id: simplePhoneId, number} = {}
            } = res;

            if (havePassword) {
                dispatch(setPasswordRequired(true));
            }

            dispatch(setCodeRequired(true));
            dispatch(setDenyResendUntil(Number(getUnixTimeStamp()) + SECONDS_TO_NEXT_CODE_SEND));
            dispatch(
                setOperation({
                    isSecure: true,
                    phoneId: securePhoneId,
                    id: secureOperationId,
                    type: OPERATION_TYPES.replace
                })
            );
            dispatch(
                setOperation({
                    isSecure: false,
                    phoneId: simplePhoneId,
                    id: simpleOperationId,
                    type: OPERATION_TYPES.replace
                })
            );
            dispatch(setCurrentPhoneId({phoneId: securePhoneId, isSecure: true}));
            dispatch(setCurrentPhoneId({phoneId: simplePhoneId, isSecure: false}));
            dispatch(setProcessedNumber(number));
            dispatch(setConfirmationCodeSentStatus(true));
        })
        .fail((error = {}) => {
            const errorCode = error.errors && Array.isArray(error.errors) && error.errors[0];

            dispatch(updateErrors({field: FIELDS_NAMES.PHONE, error: getError(FIELDS_NAMES.PHONE, errorCode)}));
        });
};

export const replaceSecurePhoneCommit = () => (dispatch, getState) => {
    const state = getState();
    const {phones: {other = [], currentSimplePhoneId} = {}, common: {csrf} = {}} = state;
    const {id} = (other.find((phone) => phone.id === currentSimplePhoneId) || {}).operation || {};

    return makeRequest(dispatch, 'yasms.secure.replace.commit', {
        id,
        csrf_token: csrf
    }).done(() => {
        dispatch(getAccountPhones());
        dispatch(onPhonesProcessEnd());
    });
};

export const phoneRequestSecureCodeSubmit = () => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf, track_id} = {}, phones: {restore = []} = {}} = state;
    const phoneId = (restore[0] || {}).id;

    return makeRequest(dispatch, 'phone-confirm-code-submit', {
        phone_id: phoneId,
        csrf_token: csrf,
        confirm_method: 'by_sms',
        isCodeWithFormat: true,
        track_id
    })
        .done((res = {}) => {
            const {deny_resend_until: denyResendUntil, is_password_required: isPasswordRequired = false} = res;

            dispatch(setPasswordRequired(isPasswordRequired));
            dispatch(setCodeRequired(true));
            dispatch(setDenyResendUntil(denyResendUntil));
            dispatch(setConfirmationCodeSentStatus(true));
        })
        .fail((error = {}) => {
            const errorCode = error.errors && Array.isArray(error.errors) && error.errors[0];

            if (errorCode === 'captcha.required') {
                dispatch(setCaptchaRequired(true));
            }
        });
};

export const sms2faCommit = () => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf, track_id} = {}, access: {isSms2faEnabled} = {}} = state;
    const {phoneCode} = getFormValues(state);

    return makeRequest(dispatch, 'yasms.sms2fa', {
        isEnabled: !isSms2faEnabled,
        csrf_token: csrf,
        code: phoneCode,
        track_id
    })
        .done((response = {}) => {
            if (response.status !== 'ok') {
                return dispatch(
                    updateErrors({
                        field: FIELDS_NAMES.PHONE_CODE,
                        error: getError(FIELDS_NAMES.PHONE_CODE, 'global')
                    })
                );
            }

            dispatch(onPhonesProcessEnd());
            dispatch(sms2faEnabled(!isSms2faEnabled));
        })
        .fail((error = {}) => {
            const code = error.errors && Array.isArray(error.errors) && error.errors[0];

            dispatch(updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: getError(FIELDS_NAMES.PHONE_CODE, code)}));
        });
};

export const enableLoginAliasSubmit = ({enableAliasAsEmail = false} = {}) => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf, track_id: trackId} = {}} = state;

    return makeRequest(dispatch, 'phones.alias.as.login.enable.submit', {
        csrf_token: csrf,
        trackId,
        enableAliasAsEmail,
        sessionid: getCookie('Session_id')
    }).done((res = {}) => {
        const {deny_resend_until: denyResendUntil, is_password_required: isPasswordRequired = false} = res;

        dispatch(setPasswordRequired(isPasswordRequired));
        dispatch(setCodeRequired(true));
        dispatch(setDenyResendUntil(denyResendUntil));
        dispatch(setConfirmationCodeSentStatus(true));
    });
};

export const enableLoginAliasCommit = () => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf, track_id} = {}} = state;
    const {phoneCode} = getFormValues(state);

    return makeRequest(dispatch, 'phones.alias.as.login.enable.commit', {
        csrf_token: csrf,
        code: phoneCode,
        track_id
    })
        .done((response = {}) => {
            if (response.status !== 'ok') {
                return dispatch(
                    updateErrors({
                        field: FIELDS_NAMES.PHONE_CODE,
                        error: getError(FIELDS_NAMES.PHONE_CODE, 'global')
                    })
                );
            }

            dispatch(getAccountPhones());
            dispatch(setIsLoginAliasState(true));
            dispatch(onPhonesProcessEnd());
        })
        .fail((error = {}) => {
            const code = error.errors && Array.isArray(error.errors) && error.errors[0];

            dispatch(updateErrors({field: FIELDS_NAMES.PHONE_CODE, error: getError(FIELDS_NAMES.PHONE_CODE, code)}));
        });
};

export const disableLoginAliasSubmit = ({enableAliasAsEmail = false} = {}) => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf, track_id: trackId} = {}, person: {havePassword} = {}} = state;

    return makeRequest(dispatch, 'phones.alias.as.login.disable.submit', {
        csrf_token: csrf,
        trackId,
        enableAliasAsEmail
    }).done(() => {
        if (!havePassword) {
            return dispatch(disableLoginAliasCommit());
        }

        dispatch(setPasswordRequired(true));
    });
};

export const disableLoginAliasCommit = () => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf, track_id} = {}, captcha = {}} = state;
    const {password, captcha: captchaAnswer} = getFormValues(state);
    const params = {
        csrf_token: csrf,
        password,
        track_id
    };

    if (captcha) {
        params.answer = captchaAnswer;
        params.key = captcha.key;
    }

    return makeRequest(dispatch, 'phones.alias.as.login.disable.commit', params)
        .done((response = {}) => {
            if (response.status !== 'ok') {
                return dispatch(
                    updateErrors({
                        field: FIELDS_NAMES.PASSWORD,
                        error: getError(FIELDS_NAMES.PHONE_CODE, 'global')
                    })
                );
            }

            dispatch(getAccountPhones());
            dispatch(onPhonesProcessEnd());
            dispatch(setIsLoginAliasState(false));
            dispatch(setIsEmailAliasState(false));
        })
        .fail((error = {}) => {
            const errorResponse = error.errors || error.error;
            const errorCode = errorResponse && Array.isArray(errorResponse) && errorResponse[0];
            const errorField = ['captcha.required', 'captcha.not_matched', 'captchalocate'].includes(errorCode)
                ? FIELDS_NAMES.CAPTCHA
                : FIELDS_NAMES.PASSWORD;

            if (errorField === 'captcha') {
                dispatch(reloadCaptcha());
            }

            if (errorCode === 'captcha.required') {
                return dispatch(setCaptchaRequired(true));
            }

            dispatch(updateErrors({field: errorField, error: getError(errorField, errorCode)}));
        });
};

export const manageEmailAlias = () => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf} = {}, phones: {restore = []} = {}} = state;
    const restorePhone = restore[0] || {};
    const {isEmailAlias} = restorePhone;
    const method = isEmailAlias ? 'yasms.alias.as.email.disable' : 'yasms.alias.as.email.enable';

    return makeRequest(dispatch, method, {
        csrf_token: csrf
    }).done(() => {
        dispatch(initTrack());
        dispatch(getAccountPhones());
        dispatch(setIsEmailAliasState(!isEmailAlias));
    });
};

export const manageNotify = (phoneId) => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf} = {}} = state;

    return makeRequest(dispatch, 'yasms.notify', {
        csrf_token: csrf,
        id: phoneId
    }).done(() => {
        dispatch(initTrack());
        dispatch(getAccountPhones());
    });
};

export const deleteSimplePhone = (phoneId) => (dispatch, getState) => {
    const state = getState();
    const {common: {csrf} = {}} = state;

    return makeRequest(dispatch, 'yasms.simple.remove', {
        csrf_token: csrf,
        id: phoneId
    }).done(() => {
        dispatch(getAccountPhones());
        dispatch(onPhonesProcessEnd());
    });
};
