import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {UniversalTile} from '@components/UniversalTile';
import {AddPhoneIcon, PlusIcon, RestorePhoneIcon, OtherPhoneIcon, PointsIcon} from '../../Icons';
import {PAGES, OPERATION_TYPES} from '../../const';

import './MainPhonesTiles.styl';

const b = cn('MainPhonesTiles');

const TEXTS = {
    addRestore: {
        title: i18n('Phones.phones.v2.main-tiles.add-restore.title'),
        description: i18n('Phones.phones.v2.main-tiles.add-restore.description')
    },
    restore: {
        descriptionPromo: i18n('Phones.phones.v2.main-tiles.restore.promo.description'),
        description: i18n('Phones.phones.v2.main-tiles.restore.description')
    },
    other: {
        descriptionPromo: i18n('Phones.phones.v2.main-tiles.other.promo.description'),
        description: i18n('Phones.phones.v2.main-tiles.other.description')
    }
};

export class MainPhonesTiles extends React.Component {
    goToPhoneBindConfirmPage = () => {
        const {goToPage} = this.props;

        goToPage({page: PAGES.phoneBindConfirm});
    };

    goToSettingsPage = (event) => {
        const {goToSettingsPage} = this.props;
        const phoneId = event && event.currentTarget.dataset.id;

        goToSettingsPage(Number(phoneId));
    };

    goToAddPhonePage = () => {
        const {goToPage} = this.props;

        goToPage({page: PAGES.addPhone});
    };

    getSecurePhoneAlert = () => {
        const {restoreOperationType, isRestoreQuarantine, quarantineEnd} = this.props;

        if (!isRestoreQuarantine) {
            return '';
        }

        if (restoreOperationType === OPERATION_TYPES.remove) {
            if (quarantineEnd <= 0) {
                return i18n('Phones.phones.v2.main-tiles.phones.quarantine.remove.soon');
            }

            return i18n({
                count: quarantineEnd,
                one: 'Phones.phones.v2.main-tiles.phones.quarantine.remove.one',
                some: 'Phones.phones.v2.main-tiles.phones.quarantine.remove.some',
                many: 'Phones.phones.v2.main-tiles.phones.quarantine.remove.many'
            }).replace('%1', quarantineEnd);
        }

        if (restoreOperationType === OPERATION_TYPES.replace) {
            if (quarantineEnd <= 0) {
                return i18n('Phones.phones.v2.main-tiles.phones.quarantine.replace.soon');
            }

            return i18n({
                count: quarantineEnd,
                one: 'Phones.phones.v2.main-tiles.phones.quarantine.replace.one',
                some: 'Phones.phones.v2.main-tiles.phones.quarantine.replace.some',
                many: 'Phones.phones.v2.main-tiles.phones.quarantine.replace.many'
            }).replace('%1', quarantineEnd);
        }
    };

    renderAddSecurePhoneTile = () => (
        <UniversalTile
            type='button'
            leftComponent={<AddPhoneIcon />}
            rightComponent={<PlusIcon />}
            title={TEXTS.addRestore.title}
            description={TEXTS.addRestore.description}
            handler={this.goToAddPhonePage}
            dataT={'phones.add-phone-tile'}
            backgroundColor={'tintSuperLight'}
            backgroundHoverColor={'tintLight'}
            isBigTitle={true}
            isBorderBox={true}
            isBigPadding={true}
        />
    );

    renderDefaultRestorePhoneTile = () => {
        const {isRestoreDefault, defaultRestorePhoneNumber, restorePhoneId} = this.props;

        return (
            <UniversalTile
                type='button'
                leftComponent={<RestorePhoneIcon />}
                rightComponent={<PointsIcon />}
                title={defaultRestorePhoneNumber}
                description={isRestoreDefault ? TEXTS.restore.descriptionPromo : TEXTS.restore.description}
                alert={this.getSecurePhoneAlert()}
                handler={this.goToSettingsPage}
                dataT={'phones.default-restore-phone'}
                dataId={restorePhoneId}
                backgroundColor={'tintSuperLight'}
                backgroundHoverColor={'tintLight'}
                isBigTitle={true}
                isBorderBox={true}
                isBigPadding={true}
            />
        );
    };

    renderOtherPhones = () => {
        const {other} = this.props;

        if (!other.length) {
            return null;
        }

        return other.map((phone) => {
            const {operation: {type} = {}, isDefault} = phone;

            return (
                type !== OPERATION_TYPES.mark && (
                    <UniversalTile
                        key={phone.number}
                        type='button'
                        leftComponent={<OtherPhoneIcon />}
                        rightComponent={<PointsIcon />}
                        title={phone.number}
                        description={isDefault ? TEXTS.other.descriptionPromo : TEXTS.other.description}
                        handler={this.goToSettingsPage}
                        dataT={'phones.default-restore-phone'}
                        dataId={phone.id}
                        backgroundColor={'tintSuperLight'}
                        backgroundHoverColor={'tintLight'}
                        isBigTitle={true}
                        isBorderBox={true}
                        isBigPadding={true}
                    />
                )
            );
        });
    };

    render() {
        const {hasRestorePhone} = this.props;

        return (
            <div className={b()}>
                {hasRestorePhone ? this.renderDefaultRestorePhoneTile() : this.renderAddSecurePhoneTile()}
                {this.renderOtherPhones()}
            </div>
        );
    }
}

MainPhonesTiles.propTypes = {
    access: PropTypes.object,
    havePassword: PropTypes.bool.isRequired,
    isLiteUser: PropTypes.bool.isRequired,
    isNeoPhonish: PropTypes.bool.isRequired,
    isSocialchik: PropTypes.bool.isRequired,
    goToPage: PropTypes.func.isRequired,
    goToSettingsPage: PropTypes.func.isRequired,
    restore: PropTypes.array.isRequired,
    other: PropTypes.array.isRequired,
    hasRestorePhone: PropTypes.bool.isRequired,
    hasOtherPhone: PropTypes.bool.isRequired,
    defaultRestorePhoneNumber: PropTypes.string,
    isRestoreDefault: PropTypes.bool,
    restoreOperationType: PropTypes.string,
    isRestoreQuarantine: PropTypes.bool,
    quarantineEnd: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
    restorePhoneId: PropTypes.oneOfType([PropTypes.string, PropTypes.number])
};
