import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {UniversalTileList} from '@components/UniversalTileList';
import {PhonesButton} from '../PhonesButton';
import {TrashIcon, ReplaceIcon, CancelIcon, RestorePhoneIcon, SmsIcon, HelpIcon} from '../../Icons';
import {PAGES, OPERATION_TYPES} from '../../const';
import {metrics} from '../../utils';

import './PhoneSettings.styl';

const b = cn('PhoneSettings');

const CANCEL_BUTTON = {
    [OPERATION_TYPES.replace]: i18n('Phones.phones.v2.settings.cancel.replace'),
    [OPERATION_TYPES.remove]: i18n('Phones.phones.v2.settings.cancel.remove')
};

const CANCEL_OPERATION_METRIC_TEXTS = {
    [OPERATION_TYPES.replace]: 'замена основного номера',
    [OPERATION_TYPES.remove]: 'удаление основного номера'
};

const TEXTS = {
    restore: {
        title: i18n('Phones.phones.v2.settings.restore.title'),
        description: i18n('Phones.phones.v2.settings.restore.description'),
        replaceButton: i18n('Phones.phones.v2.settings.restore.replace'),
        removeButton: i18n('Phones.phones.v2.settings.restore.remove'),
        howToRemoveButton: i18n('Phones.phones.v2.settings.restore.how.remove'),
        howToReplaceButton: i18n('Phones.phones.v2.settings.restore.how.replace')
    },
    other: {
        title: i18n('Phones.phones.v2.settings.other.title'),
        promoButton: i18n('Phones.phones.v2.settings.other.promo'),
        removeButton: i18n('Phones.phones.v2.settings.other.remove')
    },
    close: i18n('Phones.phones.v2.close')
};

const NOTIFICATIONS = {
    [OPERATION_TYPES.remove]: i18n('Phones.phones.v2.notification.cancel.remove'),
    [OPERATION_TYPES.replace]: i18n('Phones.phones.v2.notification.cancel.replace')
};

export class PhoneSettings extends React.Component {
    goToDeleteSecureAlertPage = () => {
        const {goToPage} = this.props;

        goToPage({page: PAGES.deleteSecureAlert});
    };

    goToReplaceSecureAlertPage = () => {
        const {goToPage} = this.props;

        goToPage({page: PAGES.replaceSecureAlert});
    };

    goToDeleteSimplePhonePage = () => {
        const {goToPage, setCurrentPhoneId, phoneId} = this.props;

        setCurrentPhoneId({phoneId, isSecure: false});
        goToPage({page: PAGES.deleteSimplePhone});
    };

    goToDeleteSecurePhonePage = () => {
        const {goToPage} = this.props;

        goToPage({page: PAGES.deleteSecurePhone});
    };

    goToReplacePhonePage = () => {
        const {goToPage} = this.props;

        goToPage({page: PAGES.replacePhone});
    };

    cancelOperation = () => {
        const {phoneId, cancelOperation, goToMainPage, restoreOperationType, setNotificationsState} = this.props;

        cancelOperation(phoneId).done(() => {
            metrics.send(['Отмена операции с карантином', CANCEL_OPERATION_METRIC_TEXTS[restoreOperationType]]);

            setNotificationsState({
                isVisible: true,
                notificationsList: [
                    {
                        text: NOTIFICATIONS[restoreOperationType]
                    }
                ]
            });
            goToMainPage();
        });
    };

    manageNotify = () => {
        const {phoneId, manageNotify, goToMainPage} = this.props;

        manageNotify(phoneId).done(() => goToMainPage());
    };

    getSecurePhoneAlert = () => {
        const {restoreOperationType, isRestoreQuarantine, quarantineEnd, phoneForReplaceSecure} = this.props;

        if (!isRestoreQuarantine) {
            return '';
        }

        if (restoreOperationType === OPERATION_TYPES.remove) {
            if (quarantineEnd <= 0) {
                return i18n('Phones.phones.v2.settings.quarantine.remove.soon');
            }

            return i18n({
                count: quarantineEnd,
                one: 'Phones.phones.v2.settings.quarantine.remove.one',
                some: 'Phones.phones.v2.settings.quarantine.remove.some',
                many: 'Phones.phones.v2.settings.quarantine.remove.many'
            }).replace('%1', quarantineEnd);
        }

        if (restoreOperationType === OPERATION_TYPES.replace) {
            if (quarantineEnd <= 0) {
                return i18n('Phones.phones.v2.settings.quarantine.replace.soon').replace('%2', phoneForReplaceSecure);
            }

            return i18n({
                count: quarantineEnd,
                one: 'Phones.phones.v2.settings.quarantine.replace.one',
                some: 'Phones.phones.v2.settings.quarantine.replace.some',
                many: 'Phones.phones.v2.settings.quarantine.replace.many'
            })
                .replace('%1', quarantineEnd)
                .replace('%2', phoneForReplaceSecure);
        }
    };

    getTiles = () => {
        const {
            isSecure,
            isRestoreQuarantine,
            restoreOperationType,
            isDefault,
            isSms2faEnabled,
            is2faEnabled,
            hasQuestion,
            isNeoPhonish,
            isSocialchik,
            isSuperLiteUser
        } = this.props;

        const canDeleteSecure =
            hasQuestion && !(is2faEnabled || isSms2faEnabled || isNeoPhonish || isSocialchik || isSuperLiteUser);

        return isSecure ? (
            <UniversalTileList
                tiles={
                    isRestoreQuarantine
                        ? [
                              {
                                  type: 'button',
                                  leftComponent: <CancelIcon />,
                                  title: CANCEL_BUTTON[restoreOperationType],
                                  handler: this.cancelOperation,
                                  dataT: `phones.settings.restore.cancel.${restoreOperationType}`,
                                  verticalAlignCenter: true,
                                  hasLeftComponentBg: true
                              }
                          ]
                        : [
                              !isDefault && {
                                  type: 'block',
                                  leftComponent: <SmsIcon />,
                                  title: TEXTS.other.promoButton,
                                  handler: this.manageNotify,
                                  dataT: 'phones.settings.simple.notify',
                                  verticalAlignCenter: true,
                                  hasLeftComponentBg: true
                              },
                              isNeoPhonish
                                  ? {
                                        type: 'button',
                                        leftComponent: <HelpIcon />,
                                        title: TEXTS.restore.howToReplaceButton,
                                        handler: this.goToReplaceSecureAlertPage,
                                        dataT: 'phones.settings.restore.how.replace',
                                        verticalAlignCenter: true,
                                        hasLeftComponentBg: true
                                    }
                                  : {
                                        type: 'button',
                                        leftComponent: <ReplaceIcon />,
                                        title: TEXTS.restore.replaceButton,
                                        handler: this.goToReplacePhonePage,
                                        dataT: 'phones.settings.restore.replace',
                                        verticalAlignCenter: true,
                                        hasLeftComponentBg: true
                                    },
                              canDeleteSecure
                                  ? {
                                        type: 'button',
                                        leftComponent: <TrashIcon />,
                                        title: TEXTS.restore.removeButton,
                                        handler: this.goToDeleteSecurePhonePage,
                                        dataT: 'phones.settings.restore.remove',
                                        verticalAlignCenter: true,
                                        hasLeftComponentBg: true
                                    }
                                  : !is2faEnabled && {
                                        type: 'button',
                                        leftComponent: <HelpIcon />,
                                        title: TEXTS.restore.howToRemoveButton,
                                        handler: this.goToDeleteSecureAlertPage,
                                        dataT: 'phones.settings.restore.how.remove',
                                        verticalAlignCenter: true,
                                        hasLeftComponentBg: true
                                    }
                          ]
                }
            />
        ) : (
            <UniversalTileList
                tiles={[
                    !isDefault && {
                        type: 'block',
                        leftComponent: <SmsIcon />,
                        title: TEXTS.other.promoButton,
                        handler: this.manageNotify,
                        dataT: 'phones.settings.simple.notify',
                        verticalAlignCenter: true,
                        hasLeftComponentBg: true
                    },
                    {
                        type: 'button',
                        leftComponent: <TrashIcon />,
                        title: TEXTS.other.removeButton,
                        handler: this.goToDeleteSimplePhonePage,
                        dataT: 'phones.settings.simple.remove',
                        verticalAlignCenter: true,
                        hasLeftComponentBg: true
                    }
                ]}
            />
        );
    };

    render() {
        const {isSecure, isRestoreQuarantine, goToMainPage, phoneNumber} = this.props;

        return (
            <div className={b()}>
                <h2 className={b('title')}>{isSecure ? TEXTS.restore.title : TEXTS.other.title}</h2>
                <div className={b('numberWrapper')}>
                    {isSecure && (
                        <div className={b('numberIcon')}>
                            <RestorePhoneIcon />
                        </div>
                    )}
                    <div className={b('number')}>{phoneNumber}</div>
                </div>
                {isSecure && <div className={b('description')}>{TEXTS.restore.description}</div>}
                {isRestoreQuarantine && isSecure && (
                    <div className={b('quarantineAlert')}>{this.getSecurePhoneAlert()}</div>
                )}
                <div className={b('tiles')}>{this.getTiles()}</div>
                <div className={b('closeButton')}>
                    <PhonesButton view='default' text={TEXTS.close} onClick={goToMainPage} />
                </div>
            </div>
        );
    }
}

PhoneSettings.propTypes = {
    havePassword: PropTypes.bool.isRequired,
    isLiteUser: PropTypes.bool.isRequired,
    isNeoPhonish: PropTypes.bool.isRequired,
    isSocialchik: PropTypes.bool.isRequired,
    isSuperLiteUser: PropTypes.bool.isRequired,
    goToPage: PropTypes.func.isRequired,
    goToMainPage: PropTypes.func.isRequired,
    cancelOperation: PropTypes.func.isRequired,
    setNotificationsState: PropTypes.func.isRequired,
    restore: PropTypes.array.isRequired,
    other: PropTypes.array.isRequired,
    hasRestorePhone: PropTypes.bool.isRequired,
    hasOtherPhone: PropTypes.bool.isRequired,
    phoneNumber: PropTypes.string,
    restoreOperationType: PropTypes.string,
    isRestoreQuarantine: PropTypes.bool,
    quarantineEnd: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
    isSecure: PropTypes.bool,
    phoneId: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
    phoneForReplaceSecure: PropTypes.string,
    manageNotify: PropTypes.func,
    isDefault: PropTypes.bool,
    setCurrentPhoneId: PropTypes.func,
    isSms2faEnabled: PropTypes.bool,
    is2faEnabled: PropTypes.bool,
    hasQuestion: PropTypes.bool
};
