import React from 'react';
import PropTypes from 'prop-types';
import {Field} from '@components/Field';
import {FIELDS_NAMES} from '@components/Field/names';
import PhoneConfirmationTimer from '@blocks/authv2/components/PhoneConfirmationTimer/PhoneConfirmationTimer.jsx';
import {classNames, getUnixTimeStamp} from '@blocks/utils';
import {Password} from '@components/Password';
import {PhonesButton} from '../PhonesButton';
import {CaptchaField} from '@components/CaptchaField';
import './PhonesCodePassConfirm.styl';

const b = classNames('PhonesCodePassConfirm');

const TEXTS = {
    codeLabel: i18n('Phones.phones.v2.confirm-code-page.code-label'),
    passLabel: i18n('Phones.phones.v2.confirm-code-page.pass-label'),
    sendNewCode: i18n('Phones.phones.v2.confirm-code-page.send-new-code')
};

class PhonesCodePassConfirm extends React.PureComponent {
    state = {
        currentTimestamp: getUnixTimeStamp(),
        code: []
    };

    static defaultProps = {
        name: FIELDS_NAMES.PHONE_CODE
    };

    ref = React.createRef();
    _timerInterval = null;

    componentDidMount() {
        this._stopTimer();
        this._startTimer();

        if (this.ref.current) {
            this.ref.current.focus();
        }
    }

    componentDidUpdate(prevProps) {
        const {denyResendUntil} = this.props;

        if (denyResendUntil !== prevProps.denyResendUntil) {
            this._updateTimestamp();
            this._startTimer();
        }
    }

    componentWillUnmount() {
        const {updateValues} = this.props;

        this.clearErrors();
        this._stopTimer();
        updateValues({field: FIELDS_NAMES.PHONE_CODE, value: ''});
    }

    _handleSubmit = (event) => {
        const {onConfirm} = this.props;

        event && event.preventDefault();
        onConfirm();
    };

    _updateTimestamp = () => this.setState({currentTimestamp: getUnixTimeStamp()});

    _startTimer = () => {
        const {denyResendUntil} = this.props;

        if (!denyResendUntil) {
            return;
        }

        this._timerInterval = setInterval(() => {
            this._updateTimestamp();

            if (getUnixTimeStamp() >= Number(denyResendUntil)) {
                this._stopTimer();
            }
        }, 1000);
    };

    _stopTimer = () => {
        if (this._timerInterval) {
            clearInterval(this._timerInterval);
            this._timerInterval = null;
        }
    };

    _getSendSMSButtonText = (showTimer) => {
        const {denyResendUntil} = this.props;
        const {currentTimestamp} = this.state;
        const text = TEXTS.sendNewCode;

        if (showTimer) {
            return (
                <>
                    {text}
                    {'  '}
                    <PhoneConfirmationTimer
                        startTime={currentTimestamp}
                        endTime={Number(denyResendUntil)}
                        isUnixTimeStamp={true}
                    />
                </>
            );
        }

        return text;
    };

    _retryToRequestPhoneConfirmationCode = () => {
        const {onRetry} = this.props;

        this._clearCode();
        onRetry();
    };

    _clearCode = () => {
        this.setState({code: []});
        const {updateValues} = this.props;

        updateValues({field: FIELDS_NAMES.PHONE_CODE, value: ''});

        if (this.ref.current) {
            this.ref.current.focus();
        }
    };

    clearErrors = () => {
        const {name, updateErrors} = this.props;

        updateErrors({field: name, error: {}});
    };

    _renderControls = () => {
        const {currentTimestamp} = this.state;
        const {denyResendUntil, isLoading, isCodeRequired} = this.props;
        const showTimer = currentTimestamp < Number(denyResendUntil);

        return (
            <div className={b('controls')}>
                <PhonesButton
                    text={i18n('_AUTH_.next')}
                    view='action'
                    onClick={this._handleSubmit}
                    size='l'
                    type='submit'
                    disabled={this._isDisabled() || isLoading}
                    width='max'
                />
                {isCodeRequired && (
                    <PhonesButton
                        onClick={this._retryToRequestPhoneConfirmationCode}
                        view='transparent'
                        type='button'
                        disabled={showTimer || isLoading}
                        size='l'
                        width='max'
                        dataT='retry-to-request-code'
                    >
                        {this._getSendSMSButtonText(showTimer)}
                    </PhonesButton>
                )}
            </div>
        );
    };

    _isDisabled = () => {
        const {error} = this.props;
        const {code} = this.state;

        return code.join('').length === 6 && !error;
    };

    _renderForm = () => {
        const {
            isWithCodeFormatExp,
            isMobile,
            isPasswordRequired,
            isCodeRequired,
            isCaptchaRequired,
            number
        } = this.props;

        return (
            <>
                {isCodeRequired && (
                    <>
                        <div className={b('text')}>{TEXTS.codeLabel.replace('%1', number)}</div>
                        <Field
                            name={FIELDS_NAMES.PHONE_CODE}
                            view='big-input'
                            size='l'
                            disabled={this._isDisabled()}
                            maxLength={isWithCodeFormatExp ? 7 : 6}
                            tInfo={'phones.poneCodeInput'}
                            options={{
                                autoComplete: 'off',
                                autoCapitalize: 'off',
                                autoCorrect: 'off',
                                inputMode: isMobile ? 'numeric' : 'text'
                            }}
                            placeholder={isWithCodeFormatExp ? '123-456' : '123456'}
                            ref={this.ref}
                            isPhoneCode={true}
                            hideLabel={true}
                        />
                    </>
                )}
                {isPasswordRequired && (
                    <>
                        <div className={b('text')}>{TEXTS.passLabel}</div>
                        <Password view='big-input' hideLabel={true} skipFocus={isCodeRequired} />
                    </>
                )}
                {isCaptchaRequired && (
                    <CaptchaField
                        isCaptchaRequired={true}
                        isShowCaptcha={true}
                        fieldView='big-input'
                        view='column'
                        centered={true}
                    />
                )}
            </>
        );
    };

    render() {
        return (
            <div className={b()}>
                <form action='#' method='POST' onSubmit={this._handleSubmit}>
                    <div className={b('body')}>{this._renderForm()}</div>

                    {this._renderControls()}
                </form>
            </div>
        );
    }
}

PhonesCodePassConfirm.propTypes = {
    name: PropTypes.string,
    number: PropTypes.string,
    code: PropTypes.string,
    error: PropTypes.string,
    onConfirm: PropTypes.func,
    onRetry: PropTypes.func.isRequired,
    denyResendUntil: PropTypes.oneOfType([PropTypes.number, PropTypes.string]),
    updateValues: PropTypes.func.isRequired,
    updateErrors: PropTypes.func.isRequired,
    setPhoneConfirmMethod: PropTypes.func.isRequired,
    isMobile: PropTypes.bool,
    isLoading: PropTypes.bool,
    isPasswordRequired: PropTypes.bool,
    isCodeRequired: PropTypes.bool,
    isCaptchaRequired: PropTypes.bool,
    isCodeConfirmed: PropTypes.bool,
    isPasswordConfirmed: PropTypes.bool,
    isWithCodeFormatExp: PropTypes.bool,
    commitConfirmation: PropTypes.func
};

export {PhonesCodePassConfirm};
