import React from 'react';
import PropTypes from 'prop-types';
import {PhonesAddPhoneLayout} from '../../layouts/PhonesAddPhoneLayout';
import {PhonesConfirmPhoneLayout} from '../../layouts/PhonesConfirmPhoneLayout';
import {PhonesConfirmCodeLayout} from '../../layouts/PhonesConfirmCodeLayout';
import {PhonesAlertLayout} from '../../layouts/PhonesAlertLayout';
import {metrics} from '../../utils';

const TEXTS = {
    tutorial: {
        title: i18n('Phones.phones.v2.tutorial.quarantine.title'),
        description: i18n('Phones.phones.v2.tutorial.quarantine.description'),
        buttonTutorialOk: i18n('Phones.phones.v2.tutorial.ok')
    }
};

const NOTIFICATIONS = {
    replace: i18n('Phones.phones.v2.notification.replace')
};

export class ReplacePhonePage extends React.Component {
    state = {
        showTutorial: false
    };

    manageTutorial = () => {
        this.setState({showTutorial: !this.state.showTutorial});
    };

    openProfileDevices = () => {
        window.open('/profile/devices', '_blank');
    };

    getButtonsTutorial = () => {
        return [
            {
                text: TEXTS.tutorial.buttonTutorialOk,
                view: 'action',
                onClick: this.manageTutorial
            }
        ];
    };

    setReplaceAddPhoneStep = () => {
        const {setReplaceAddPhoneStep} = this.props;

        setReplaceAddPhoneStep(true);
    };

    commitConfirmationSecure = () => {
        const {
            checkCode,
            checkPassword,
            setReplaceSecondStep,
            setPasswordRequired,
            setCaptchaRequired,
            clearFields,
            havePassword
        } = this.props;
        const handlers = [checkCode()];

        if (havePassword) {
            handlers.push(checkPassword());
        }

        Promise.all(handlers)
            .then(() => {
                setReplaceSecondStep(true);
                setPasswordRequired(false);
                setCaptchaRequired(false);
                clearFields();
            })
            .catch(() => {
                // do nothing
            });
    };

    commitConfirmationSimple = () => {
        const {checkCode, replaceSecurePhoneCommit, goToMainPage, setNotificationsState} = this.props;

        checkCode({isSecure: false}).done(() =>
            replaceSecurePhoneCommit().done(() => {
                metrics.send(['Заменен основной телефон']);
                setNotificationsState({
                    isVisible: true,
                    notificationsList: [
                        {
                            text: NOTIFICATIONS.replace,
                            iconType: 'ok'
                        }
                    ]
                });
                goToMainPage();
            })
        );
    };

    commitConfirmationNotPhoneAccess = () => {
        const {checkCode, checkPassword, replaceSecurePhoneCommit, goToMainPage} = this.props;

        Promise.all([checkCode({isSecure: false}), checkPassword({isSecure: false})]).then(() =>
            replaceSecurePhoneCommit().done(() => {
                metrics.send(['Запущена замена основного телефона с карантином']);
                goToMainPage();
            })
        );
    };

    resendSimpleCode = () => {
        const {resendCode} = this.props;

        resendCode({isSecure: false});
    };

    render() {
        const {
            isModal,
            processedNumber,
            isConfirmationCodeSent,
            replaceSecurePhoneSubmit,
            resendCode,
            numberRestore,
            isSecondReplaceStep,
            isReplaceAddPhoneStep,
            hasPhoneAccess
        } = this.props;
        const {showTutorial} = this.state;

        if (showTutorial) {
            return (
                <PhonesAlertLayout
                    decor='tutorial'
                    title={TEXTS.tutorial.title}
                    text={TEXTS.tutorial.description}
                    textAsDangerouslySet={true}
                    isModal={isModal}
                    buttons={this.getButtonsTutorial()}
                    onBackButtonClick={this.manageTutorial}
                />
            );
        }

        if (isSecondReplaceStep) {
            return (
                <PhonesConfirmCodeLayout
                    isModal={isModal}
                    commitConfirmation={this.commitConfirmationSimple}
                    resendCode={this.resendSimpleCode}
                    number={processedNumber}
                />
            );
        }

        if (isConfirmationCodeSent) {
            return (
                <PhonesConfirmCodeLayout
                    isModal={isModal}
                    commitConfirmation={
                        hasPhoneAccess ? this.commitConfirmationSecure : this.commitConfirmationNotPhoneAccess
                    }
                    resendCode={hasPhoneAccess ? resendCode : this.resendSimpleCode}
                    number={hasPhoneAccess ? numberRestore : processedNumber}
                />
            );
        }

        if (isReplaceAddPhoneStep) {
            return (
                <PhonesAddPhoneLayout place='default' isModal={isModal} confirmationSubmit={replaceSecurePhoneSubmit} />
            );
        }

        return (
            <PhonesConfirmPhoneLayout
                place='replace'
                decor='replace'
                isModal={isModal}
                onSubmit={this.setReplaceAddPhoneStep}
                onWarningClick={this.manageTutorial}
            />
        );
    }
}

ReplacePhonePage.propTypes = {
    isModal: PropTypes.bool,
    error: PropTypes.string,
    checkCode: PropTypes.func,
    checkPassword: PropTypes.func,
    goToMainPage: PropTypes.func,
    setNotificationsState: PropTypes.func,
    isConfirmationCodeSent: PropTypes.bool,
    numberRestore: PropTypes.string,
    processedNumber: PropTypes.string,
    replaceSecurePhoneCommit: PropTypes.func.isRequired,
    replaceSecurePhoneSubmit: PropTypes.func.isRequired,
    resendCode: PropTypes.func,
    setReplaceSecondStep: PropTypes.func.isRequired,
    isSecondReplaceStep: PropTypes.bool,
    setPasswordRequired: PropTypes.func.isRequired,
    setCaptchaRequired: PropTypes.func.isRequired,
    hasPhoneAccess: PropTypes.bool,
    isReplaceAddPhoneStep: PropTypes.bool,
    setReplaceAddPhoneStep: PropTypes.func,
    clearFields: PropTypes.func,
    havePassword: PropTypes.bool
};
