import React from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {cn} from '@bem-react/classname';
import {Header} from './components/Header';
import {InfoIcon} from './components/InfoIcon';
import {DefaultAvatarIcon} from './components/DefaultAvatarIcon';
import metrics from '@blocks/metrics';
import {
    WEB_CARD_LOGIN,
    WEB_CARD_LOGIN_SHOWN,
    WEB_CARD_LOGIN_CLOSED,
    WEB_CARD_LOGIN_NOT_ME,
    WEB_CARD_LOGIN_CHANGE_PASSWORD,
    WEB_CARD_LOGIN_OK,
    WEB_CARD_LOGIN_BACK,
    WEB_CARD_LOGIN_ERROR,
    WEB_CARD_LOGIN_LOGOUT
} from '@blocks/pushes/metrics_constants';
import './ActivityAlert.styl';
const b = cn('ActivityAlert');

const GLOBAL_LOGOUT_LINK = '/passport?origin=passport_profile&mode=logout&global=1';

const TITLES = {
    default: i18n('Frontend.push.activity-alert-title'),
    changed_password: i18n('Frontend.push.activity-alert-title.changed-password'),
    collector_add: i18n('Frontend.push.activity-alert-title.collector-add'),
    app_password_add: i18n('Frontend.push.activity-alert-title.app-password-add'),
    login_method_change: i18n('Frontend.push.activity-alert-title.login-method-change'),
    email_change: i18n('Frontend.push.activity-alert-title.email-change'),
    card_change: i18n('Frontend.push.activity-alert-title.card-change'),
    hint_change: i18n('Frontend.push.activity-alert-title.hint-change'),
    social_allow: i18n('Frontend.push.activity-alert-title.social-allow'),
    social_add: i18n('Frontend.push.activity-alert-title.social-add'),
    phone_change: i18n('Frontend.push.activity-alert-title.phone-change'),
    restore: i18n('Frontend.push.activity-alert-title.restore')
};

class ActivityAlert extends React.PureComponent {
    componentDidMount() {
        const {isAccountError, finishErrorUrl} = this.props;

        if (isAccountError) {
            metrics.send([WEB_CARD_LOGIN, WEB_CARD_LOGIN_ERROR]);
            window.location.href = finishErrorUrl;
            return;
        }

        metrics.send([WEB_CARD_LOGIN, WEB_CARD_LOGIN_SHOWN]);
    }

    _showNotMe = () => {
        const {showNotMe} = this.props;

        metrics.send([WEB_CARD_LOGIN, WEB_CARD_LOGIN_NOT_ME]);
        showNotMe();
    };

    _showMain = () => {
        const {showMain} = this.props;

        metrics.send([WEB_CARD_LOGIN, WEB_CARD_LOGIN_BACK]);
        showMain();
    };

    _beginChangePasswordFlow = () => {
        const {beginChangePasswordFlow} = this.props;

        metrics.send([WEB_CARD_LOGIN, WEB_CARD_LOGIN_CHANGE_PASSWORD]);
        beginChangePasswordFlow();
    };

    _logout = () => {
        const {yu, finishOkUrl} = this.props;

        metrics.send([WEB_CARD_LOGIN, WEB_CARD_LOGIN_LOGOUT]);
        window.location.href = `${GLOBAL_LOGOUT_LINK}&yu=${yu}&retpath=${finishOkUrl}`;
    };

    _closePopup = () => {
        const {finishCancelUrl} = this.props;

        metrics.send([WEB_CARD_LOGIN, WEB_CARD_LOGIN_CLOSED]);
        window.location.href = finishCancelUrl;
    };

    _approve = () => {
        const {finishOkUrl} = this.props;

        metrics.send([WEB_CARD_LOGIN, WEB_CARD_LOGIN_OK]);
        window.location.href = finishOkUrl;
    };

    _renderMainTitle = () => {
        const {login, name, havePassword, activityAlert} = this.props;
        const {titleKey = 'default'} = activityAlert;
        const shownLogin = havePassword ? login : name;

        return (
            <h2
                className={b('title')}
                dangerouslySetInnerHTML={{
                    __html: TITLES[titleKey].replace('%login', `<span class="${b('login')}">${shownLogin}</span>`)
                }}
            />
        );
    };

    _renderMain = () => {
        const {isDefaultAvatar, avatarLink, activityAlert} = this.props;
        const {isShowNotMe, mapUrl, location, date, browserName} = activityAlert;

        const avatarStyle = {
            backgroundImage: avatarLink
        };

        return (
            <div className={b('content')}>
                <Header showMain={this._showMain} closePopup={this._closePopup} b={b} isShowNotMe={isShowNotMe} />

                <div className={b('mapWrapper', {isNoMap: !mapUrl})}>
                    {mapUrl && <div className={b('yamap')} style={{backgroundImage: `url('${mapUrl}')`}} />}
                    <div className={b('avatarWrapper')}>
                        {isDefaultAvatar ? (
                            <DefaultAvatarIcon className={b('defaultAvatar', {isNoMap: !mapUrl})} />
                        ) : (
                            <span className={b('avatar', {isNoMap: !mapUrl})} style={avatarStyle} />
                        )}
                    </div>
                </div>

                {this._renderMainTitle()}

                <div className={b('loginInfo')}>
                    {location && <div className={b('loginInfoItem')}>{location}</div>}
                    {(date || browserName) && (
                        <div className={b('loginInfoItem')}>
                            {date && browserName && `${date}, ${browserName}`}
                            {date && !browserName && `${date}`}
                            {!date && browserName && `${browserName}`}
                        </div>
                    )}
                </div>

                <div className={b('footerButtons')}>
                    <Button type='button' view='action' size='l' onClick={this._closePopup}>
                        {i18n('Frontend.push.activity-alert-it-is-me')}
                    </Button>
                    <Button type='button' view='default' size='l' onClick={this._showNotMe}>
                        <InfoIcon className={b('footerButtonIcon')} />
                        {i18n('Frontend.push.activity-alert-not-me')}
                    </Button>
                </div>
            </div>
        );
    };

    _renderNotMe = () => {
        const {havePassword, activityAlert} = this.props;
        const {isShowNotMe} = activityAlert;

        return (
            <div className={b('content')}>
                <Header showMain={this._showMain} closePopup={this._closePopup} b={b} isShowNotMe={isShowNotMe} />

                <div className={b('separator')} />

                <h2 className={b('title')}>{i18n('Frontend.push.activity-alert-title2')}</h2>

                <div className={b('notMeInfo')}>
                    <div className={b('notMeInfoItem')}>{i18n('Frontend.push.activity-alert-not-me-info-for-all')}</div>
                    {havePassword ? (
                        <div className={b('notMeInfoItem')}>
                            {i18n('Frontend.push.activity-alert-not-me-info-have-pas')}
                        </div>
                    ) : (
                        <div className={b('notMeInfoItem')}>
                            {i18n('Frontend.push.activity-alert-not-me-info-no-pas')}
                        </div>
                    )}
                </div>

                {havePassword ? (
                    <div data-t='push_activity_alert_change_password' className={b('button')}>
                        <Button
                            type='button'
                            view='action'
                            size='l'
                            width='max'
                            onClick={this._beginChangePasswordFlow}
                        >
                            {i18n('Frontend.push.activity-alert-change-password')}
                        </Button>
                    </div>
                ) : (
                    <div data-t='push_activity_alert_logout' className={b('button')}>
                        <Button type='button' view='action' size='l' width='max' onClick={this._logout}>
                            {i18n('Frontend.push.activity-alert-logout')}
                        </Button>
                    </div>
                )}

                <div data-t='push_activity_alert_approve' className={b('button', {marginTop: true})}>
                    <Button type='button' view='transparent' size='l' width='max' onClick={this._approve}>
                        {i18n('Frontend.push.activity-alert-approve')}
                    </Button>
                </div>
            </div>
        );
    };

    render() {
        const {isAccountError, activityAlert = {}} = this.props;
        const {isShowNotMe} = activityAlert;

        if (isAccountError) {
            return null;
        }

        return <div className={b('wrapper')}>{isShowNotMe ? this._renderNotMe() : this._renderMain()}</div>;
    }
}

ActivityAlert.propTypes = {
    name: PropTypes.string,
    isDefaultAvatar: PropTypes.bool,
    avatarLink: PropTypes.string,
    havePassword: PropTypes.bool,
    activityAlert: PropTypes.object.isRequired,
    isAccountError: PropTypes.bool,
    beginChangePasswordFlow: PropTypes.func,
    finishCancelUrl: PropTypes.string.isRequired,
    finishErrorUrl: PropTypes.string.isRequired,
    finishOkUrl: PropTypes.string.isRequired,
    showMain: PropTypes.func.isRequired,
    showNotMe: PropTypes.func.isRequired,
    login: PropTypes.string,
    yu: PropTypes.string
};

export {ActivityAlert};
