import React from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import {withResize} from '../hocs/withResize';
import {Button} from '@components/Button';
import {Header} from '@blocks/authv2/components/Layout/Header';
import {CloseButton} from '@blocks/pushes/components/CloseButton/CloseButton';
import {cn} from '@bem-react/classname';
import metrics from '@blocks/metrics';
import {
    WEB_PUSH,
    WEB_PUSH_SHOWN,
    WEB_PUSH_CHANGE_PASSWORD,
    WEB_PUSH_ERROR,
    WEB_PUSH_CLOSED
} from '@blocks/pushes/metrics_constants';
import {beginChangePasswordFlow, setPopupSize} from '@blocks/authv2/actions/nativeMobileApi';
import './GetCode.styl';
const b = cn('GetCode');

class GetCode extends React.PureComponent {
    componentDidMount() {
        const {getCode: {isError} = {}} = this.props;

        isError ? metrics.send([WEB_PUSH, WEB_PUSH_ERROR]) : metrics.send([WEB_PUSH, WEB_PUSH_SHOWN]);
    }

    componentDidUpdate(prevProps) {
        const {
            getCode: {isError}
        } = this.props;

        if (prevProps.getCode.isError !== isError) {
            metrics.send([WEB_PUSH, WEB_PUSH_ERROR]);
        }
    }

    _closePopup = () => {
        const {
            getCode: {isError},
            am: {finishOkUrl, finishErrorUrl}
        } = this.props;

        if (isError) {
            window.location = finishErrorUrl;
            return;
        }

        metrics.send([WEB_PUSH, WEB_PUSH_CLOSED]);
        window.location = finishOkUrl;
    };

    _beginChangePasswordFlow = () => {
        const {beginChangePasswordFlow} = this.props;

        metrics.send([WEB_PUSH, WEB_PUSH_CHANGE_PASSWORD]);
        beginChangePasswordFlow();
    };

    _formattingCode = (code) => {
        if (code.length === 6) {
            const arr = code.split('');

            arr.splice(3, 0, ' ');

            return arr.join('');
        }
        return code;
    };

    _renderError = () => {
        return (
            <div className={b('content')}>
                <div className={b('errorTitle')}>{i18n('Frontend.auth.get-code-error-title')}</div>
                <div className={b('errorDescr')}>{i18n('Frontend.auth.get-code-error-description')}</div>
                <div data-t='push_submit_qr_err' className={b('errorButton')}>
                    <Button type='button' view='action' size='l' width='max' onClick={this._closePopup}>
                        {i18n('Dashboard.ok')}
                    </Button>
                </div>
            </div>
        );
    };

    _renderCard = () => {
        const {getCode: {code} = {}} = this.props;

        return (
            <div className={b('content')}>
                <Header />
                <div className={b('close')}>
                    <CloseButton closeHandler={this._closePopup} mode='relative' />
                </div>

                {code && (
                    <>
                        <div className={b('code')}>{this._formattingCode(code)}</div>
                        <div className={b('codeDescr')}>{i18n('Frontend.auth.code-descr')}</div>
                    </>
                )}

                <div className={b('buttonDescr')}>
                    {i18n('Frontend.auth.button-descr')}
                    <button
                        data-t='push_change_password'
                        className={b('button')}
                        onClick={this._beginChangePasswordFlow}
                        type='button'
                    >
                        {i18n('Frontend.auth.change-password')}
                    </button>
                </div>
            </div>
        );
    };

    render() {
        const {getCode: {isError} = {}} = this.props;

        return <div className={b('wrapper')}>{isError ? this._renderError() : this._renderCard()}</div>;
    }
}

const mapStateToProps = (state = {}) => {
    const {getCode = {}, am = {}} = state;

    return {
        getCode,
        am
    };
};

const mapDispatchToProps = {
    beginChangePasswordFlow,
    setPopupSize
};

GetCode.propTypes = {
    am: PropTypes.object.isRequired,
    getCode: PropTypes.object.isRequired,
    beginChangePasswordFlow: PropTypes.func.isRequired
};

const connectedGetCode = connect(mapStateToProps, mapDispatchToProps)(withResize(GetCode));

export {connectedGetCode as GetCode};
