import React from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {CustomsBackground} from '@components/CustomsBackground';
import {Icon} from '@components/IconLego';
import {BrokenIdIcon} from '@blocks/Components/BrokenIdIcon';
import {Spin} from '@components/Spin';
import {cn} from '@bem-react/classname';
import metrics from '@blocks/metrics';
import {
    WEB_UNILINK,
    WEB_UNILINK_SHOWN,
    WEB_UNILINK_CHANGE_ACCOUNT,
    WEB_UNILINK_ERROR
} from '@blocks/pushes/metrics_constants';
import {AccountSelect} from '@blocks/Components/AccountSelect';
import {Logo} from '@blocks/pushes/icons/Logo/Logo';
import {CloseButton} from '@blocks/pushes/components/CloseButton/CloseButton';
import {ConfirmSlider} from './components/ConfirmSlider';
import './QrSecureAuth.styl';
const b = cn('QrSecureAuth');

const CHOSE_ACCOUNT_MASK = 99;
const TIMER_VALUE = 3000;

const DEVICE_NAMES = {
    ya_smart_device_device_name: i18n('_AUTH_.push.qr_secure_auth.device_name.ya_smart_device')
};

class QrSecureAuth extends React.PureComponent {
    state = {
        isButtonPressing: false
    };

    componentDidMount() {
        const {isAccountError, qrSecureAuth = {}} = this.props;
        const {isTrackError} = qrSecureAuth;
        const isError = isAccountError || isTrackError;

        isError ? metrics.send([WEB_UNILINK, WEB_UNILINK_ERROR]) : metrics.send([WEB_UNILINK, WEB_UNILINK_SHOWN]);
    }

    componentDidUpdate(prevProps) {
        const {qrSecureAuth} = this.props;
        const {isAuthPrepareWithCredError} = qrSecureAuth;

        if (prevProps.qrSecureAuth.isAuthPrepareWithCredError !== isAuthPrepareWithCredError) {
            metrics.send([WEB_UNILINK, WEB_UNILINK_ERROR]);
        }
    }

    _handleButtonPress = (event) => {
        event.preventDefault();
        event.stopPropagation();
        const {authPrepareWithCred, isTouch} = this.props;

        if (isTouch) {
            this.setState({isButtonPressing: true});
        }

        this.buttonPressTimer = setTimeout(() => {
            this.setState({isButtonPressing: false});
            authPrepareWithCred();
        }, TIMER_VALUE);
    };

    _handleButtonRelease = () => {
        this.setState({isButtonPressing: false});
        clearTimeout(this.buttonPressTimer);
    };

    _goToYandex = () => {
        const {settings = {}} = this.props;
        const {tld = 'ru'} = settings;

        location.href = `//yandex.${tld}`;
    };

    _closePopup = () => {
        const {isAccountError, qrSecureAuth = {}, am = {}} = this.props;
        const {finishCancelUrl, finishErrorUrl} = am;
        const {isAuthPrepareWithCredError} = qrSecureAuth;

        if (isAccountError || isAuthPrepareWithCredError) {
            window.location = finishErrorUrl;
            return;
        }
        window.location = finishCancelUrl;
    };

    _openAccountsList = () => {
        const {hasUnitedAccounts, chooseAccount, authHref, am = {}} = this.props;
        const {isAm} = am;

        if (hasUnitedAccounts) {
            metrics.send([WEB_UNILINK, WEB_UNILINK_CHANGE_ACCOUNT]);
            isAm ? chooseAccount(CHOSE_ACCOUNT_MASK) : (window.location = authHref);
        }
    };

    _renderCard = () => {
        const {isButtonPressing} = this.state;
        const {
            qrSecureAuth = {},
            am = {},
            name,
            avatarLink,
            isTouch,
            isSliderConfirmationDesign,
            authPrepareWithCred,
            publicDisplayName
        } = this.props;
        const {isAm} = am;
        const {osFamily, regionName, deviceName} = qrSecureAuth;
        const showPushTitle = osFamily || regionName || deviceName;

        const avatarStyle = {
            backgroundImage: avatarLink
        };

        if (isSliderConfirmationDesign) {
            return (
                <div className={b('content')}>
                    <div className={b('header')}>
                        <Logo className={b('headerLogo')} />
                        {isAm && <CloseButton closeHandler={this._closePopup} />}
                    </div>
                    <AccountSelect onClick={this._openAccountsList} className={b('accountSelect')} />
                    <div className={b('text')}>
                        {publicDisplayName ? (
                            <span>
                                {publicDisplayName},{' '}
                                {deviceName
                                    ? i18n('Frontend.push.qr_secure_auth.confirm_postfix')
                                    : i18n('Frontend.push.qr_secure_auth.confirm_postfix.with_unknown_device')}
                            </span>
                        ) : deviceName ? (
                            i18n('Frontend.push.qr_secure_auth.confirm')
                        ) : (
                            i18n('Frontend.push.qr_secure_auth.confirm.with_unknown_device')
                        )}
                        <br />
                        <div className={b('textFrom')}>
                            <span>
                                {[
                                    DEVICE_NAMES[deviceName] ||
                                        deviceName ||
                                        i18n('Frontend.push.qr_secure_auth.unknown_device'),
                                    osFamily,
                                    regionName
                                ]
                                    .filter(Boolean)
                                    .join(', ')}
                            </span>
                        </div>
                    </div>
                    <ConfirmSlider className={b('confirmSlider')} onConfirm={authPrepareWithCred} />
                </div>
            );
        }

        return (
            <div className={b('content')}>
                {isAm && (
                    <button className={b('close')} onClick={this._closePopup}>
                        <Icon size='xs' glyph='type-cross-websearch' className={b('closeIcon')} />
                    </button>
                )}

                <div className={b('avatarWrapper')}>
                    <span className={b('avatar')} style={avatarStyle} />
                </div>

                <div className={b('userName')}>{name}</div>

                {showPushTitle && <div className={b('title')}>{i18n('Frontend.auth.qr-auth-push-title')}</div>}

                <div className={b('enterInfo')}>
                    {regionName && <div className={b('enterInfoItem')}>{regionName}</div>}
                    {osFamily && <div className={b('enterInfoItem')}>{osFamily}</div>}
                    {deviceName && <div className={b('enterInfoItem')}>{DEVICE_NAMES[deviceName] || deviceName}</div>}
                </div>

                <div data-t='push_submit_qr' className={b('submitButton', {isButtonPressing, isNoTouch: !isTouch})}>
                    <Button
                        type='button'
                        view='action'
                        size='l'
                        width='max'
                        onTouchStart={this._handleButtonPress}
                        onTouchEnd={this._handleButtonRelease}
                        onMouseDown={this._handleButtonPress}
                        onMouseUp={this._handleButtonRelease}
                        onMouseLeave={this._handleButtonRelease}
                    >
                        {i18n('Frontend.auth.qr-auth-push-submit')}
                    </Button>
                </div>

                <div data-t='push_open_account_list' className={b('carouselButton', {isNoTouch: !isTouch})}>
                    <Button type='button' view='action' size='l' width='max' onClick={this._openAccountsList}>
                        {i18n('Frontend.auth.enter-another-account')}
                    </Button>
                </div>
            </div>
        );
    };

    _getErrorTitle = () => {
        const {qrSecureAuth: {isTrackError, errorText} = {}, is2faEnabled} = this.props;

        if (isTrackError) {
            return i18n('Frontend.auth.qr-secure-track-error-title');
        }

        if (errorText === 'account.invalid_type' && is2faEnabled) {
            return i18n('Frontend.auth.qr-secure-2fa-error-title');
        } else {
            return i18n('Frontend.auth.fail-enter');
        }
    };

    _getErrorDescr = () => {
        const {qrSecureAuth: {isTrackError, errorText} = {}, is2faEnabled} = this.props;

        if (isTrackError) {
            return i18n('Frontend.auth.qr-secure-track-error-description');
        }

        if (errorText === 'account.invalid_type' && is2faEnabled) {
            return i18n('Frontend.auth.qr-secure-2fa-error-description');
        } else {
            return i18n('Frontend.auth.enter-other-device');
        }
    };

    _renderError = () => {
        const {am: {isAm} = {}, isSliderConfirmationDesign} = this.props;

        if (isSliderConfirmationDesign) {
            return (
                <div className={b('content')}>
                    <div className={b('header')}>
                        <Logo className={b('headerLogo')} />
                        {isAm && <CloseButton closeHandler={this._closePopup} />}
                    </div>
                    <BrokenIdIcon className={b('errorIcon')} />
                    <div
                        className={b('errorTitle', {
                            sliderConfirmationDesign: true
                        })}
                    >
                        {this._getErrorTitle()}
                    </div>
                    <div
                        className={b('errorDescr', {
                            sliderConfirmationDesign: true
                        })}
                    >
                        {this._getErrorDescr()}
                    </div>
                </div>
            );
        }

        return (
            <div className={b('content')}>
                <div className={b('errorTitle')}>{this._getErrorTitle()}</div>
                <div className={b('errorDescr')}>{this._getErrorDescr()}</div>
                {isAm ? (
                    <div data-t='push_submit_qr_err' className={b('errorButton')}>
                        <Button type='button' view='action' size='l' width='max' onClick={this._closePopup}>
                            {i18n('Dashboard.ok')}
                        </Button>
                    </div>
                ) : (
                    <div className={b('errorButton')}>
                        <Button
                            type='submit'
                            text={i18n('OneDomik.device.success')}
                            view='action'
                            width='max'
                            size='l'
                            onClick={this._goToYandex}
                        />
                    </div>
                )}
            </div>
        );
    };

    _renderSuccessScreen = () => {
        return (
            <div className={b('content')}>
                <div className={b('successTitle')}>{i18n('OneDomik.device.enter.success')}</div>

                <div className={b('successButton')}>
                    <Button
                        type='submit'
                        text={i18n('OneDomik.device.success')}
                        view='action'
                        width='max'
                        size='l'
                        onClick={this._goToYandex}
                    />
                </div>
            </div>
        );
    };

    render() {
        const {isAccountError, qrSecureAuth = {}, isSliderConfirmationDesign, am = {}} = this.props;
        const {isAuthPrepareWithCredError, isLoading, isShowSuccessScreen, isTrackError} = qrSecureAuth;
        const isError = isAuthPrepareWithCredError || isAccountError || isTrackError;
        const {isAm} = am;

        return (
            <>
                <div
                    className={b('wrapper', {
                        modalDesign: isSliderConfirmationDesign && !isAm,
                        isSliderConfirmationDesign
                    })}
                >
                    {!isAm && <CustomsBackground />}
                    {!isError && !isShowSuccessScreen && this._renderCard()}
                    {!isError && isShowSuccessScreen && this._renderSuccessScreen()}
                    {isError && this._renderError()}
                    {isLoading && (
                        <>
                            <div className={b('loaderOverlay')} />
                            <Spin className={b('loader')} progress={true} size='l' />
                        </>
                    )}
                </div>
            </>
        );
    }
}

QrSecureAuth.propTypes = {
    settings: PropTypes.object.isRequired,
    isSliderConfirmationDesign: PropTypes.bool,
    am: PropTypes.object.isRequired,
    name: PropTypes.string,
    publicDisplayName: PropTypes.string,
    avatarLink: PropTypes.string,
    qrSecureAuth: PropTypes.object.isRequired,
    chooseAccount: PropTypes.func.isRequired,
    primaryActionTriggered: PropTypes.func.isRequired,
    hasUnitedAccounts: PropTypes.bool,
    isAccountError: PropTypes.bool,
    authPrepareWithCred: PropTypes.func.isRequired,
    authHref: PropTypes.string,
    isTouch: PropTypes.bool.isRequired,
    is2faEnabled: PropTypes.bool
};

export {QrSecureAuth};
