import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import throttle from 'lodash/throttle';
import {persistSyntheticEvent} from '@blocks/utils';
import {RightArrowIcon} from '../RightArrowIcon';
import './ConfirmSlider.styl';

const b = cn('ConfirmSlider');
const MOVE_WITH_ANIMATION_SPEED = 0.3; // pixels per millisecond

export class ConfirmSlider extends React.PureComponent {
    static propTypes = {
        onConfirm: PropTypes.func.isRequired,
        className: PropTypes.string
    };

    _root = React.createRef();
    _inside = React.createRef();
    _insideBg = React.createRef();
    _insideIcon = React.createRef();

    _isTouched = false;

    state = {
        isComplete: false,
        onStartPosition: true,
        jiggleIn: false,
        jiggleOut: false,
        moveWithAnimation: false
    };

    get _insideIconWidth() {
        return this._insideIcon.current.getBoundingClientRect().width;
    }

    componentDidMount() {
        this._startJiggleAnimation();
    }

    componentDidUpdate(prevProps, prevState) {
        if (this.state.onStartPosition && !prevState.onStartPosition) {
            this._startJiggleAnimation();
        }
    }

    _startJiggleAnimation = () => {
        setTimeout(() => {
            if (!this.state.onStartPosition) {
                return;
            }

            this.setState({jiggleIn: true, jiggleOut: false});
            setTimeout(() => {
                if (!this.state.onStartPosition) {
                    return;
                }

                this.setState({jiggleIn: false, jiggleOut: true});
                this._startJiggleAnimation();
            }, 1400);
        }, 2200);
    };

    _onTouchStart = (event) => {
        if (this.state.isComplete || this.state.moveWithAnimation) {
            return;
        }
        this._isTouched = true;

        const touch = event.changedTouches ? event.changedTouches[0] : event;
        const rect = this._root.current.getBoundingClientRect();
        const insideTouchPositionX = touch.clientX - rect.left;

        const currentWidth = Math.max(
            this._inside.current.getBoundingClientRect().width,
            this._insideIcon.current.getBoundingClientRect().width
        );

        this._touchStartAdditionalWidth = currentWidth - insideTouchPositionX;

        this.setState({onStartPosition: false});
    };

    _onTouchMove = persistSyntheticEvent(
        throttle((event) => {
            if (this.state.isComplete || !this._isTouched) {
                return;
            }

            const touch = event.changedTouches ? event.changedTouches[0] : event;
            const rootRect = this._root.current.getBoundingClientRect();
            const insideTouchPositionX = touch.clientX - rootRect.left;
            const width = this._touchStartAdditionalWidth + insideTouchPositionX;

            this._inside.current.style.width = `${width}px`;

            const opacity = (width - this._insideIconWidth) / (rootRect.width - this._insideIconWidth);

            this._insideBg.current.style.opacity = Math.max(0, Math.min(1, opacity));

            if (width >= rootRect.width) {
                this._complete();
            }
        })
    );

    _onTouchEnd = () => {
        if (this.state.isComplete || !this._isTouched) {
            return;
        }

        const rootRect = this._root.current.getBoundingClientRect();
        const insideRect = this._inside.current.getBoundingClientRect();
        const rootWidth = rootRect.width - this._insideIconWidth;
        const insideWidth = insideRect.width - this._insideIconWidth;

        if (insideWidth >= rootWidth) {
            this._complete();
        } else {
            this.setState({moveWithAnimation: true});
            if (insideWidth / rootWidth >= 0.8) {
                const time = (rootWidth - insideWidth) / MOVE_WITH_ANIMATION_SPEED;

                this._inside.current.style['transition-duration'] = `${time}ms`;
                this._insideBg.current.style['transition-duration'] = `${time}ms`;
                this._inside.current.style.width = '100%';
                this._insideBg.current.style.opacity = '1';
                setTimeout(() => {
                    this._complete();
                }, time);
            } else {
                const time = insideWidth / MOVE_WITH_ANIMATION_SPEED;

                this._inside.current.style['transition-duration'] = `${time}ms`;
                this._insideBg.current.style['transition-duration'] = `${time}ms`;
                this._inside.current.style.width = `${this._insideIconWidth}px`;
                this._insideBg.current.style.opacity = '0';
                setTimeout(() => {
                    this._inside.current.style['transition-duration'] = null;
                    this._insideBg.current.style['transition-duration'] = null;
                    this.setState({moveWithAnimation: false, onStartPosition: true});
                }, time);
            }
        }

        this._isTouched = false;
    };

    _complete() {
        this.setState({isComplete: true});
        setTimeout(() => {
            this.props.onConfirm();
        }, 600);
    }

    render() {
        const {className} = this.props;

        return (
            <div
                className={b('', [className])}
                onMouseMove={this._onTouchMove}
                onTouchMove={this._onTouchMove}
                onTouchEnd={this._onTouchEnd}
                onMouseLeave={this._onTouchEnd}
                onMouseUp={this._onTouchEnd}
                ref={this._root}
            >
                <div className={b('defaultText')}>
                    {i18n('Frontend.push.qr_secure_auth.confirm_slider.default_text')}
                </div>
                <div
                    className={b('inside', {
                        withAnimation: this.state.moveWithAnimation
                    })}
                    ref={this._inside}
                >
                    <div className={b('insideBg')} ref={this._insideBg} />
                    <div
                        className={b('acceptedText', {
                            show: this.state.isComplete
                        })}
                    >
                        {i18n('Frontend.push.qr_secure_auth.confirm_slider.accepted')}
                    </div>
                    <div
                        className={b('insideIcon', {
                            jiggleIn: this.state.onStartPosition && this.state.jiggleIn,
                            jiggleOut: this.state.onStartPosition && this.state.jiggleOut
                        })}
                        onMouseDown={this._onTouchStart}
                        onTouchStart={this._onTouchStart}
                        ref={this._insideIcon}
                    >
                        <RightArrowIcon className={b('rightIcon')} />
                    </div>
                </div>
            </div>
        );
    }
}
