import React from 'react';
import {connect} from 'react-redux';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import mapStateToProps from './mapStateToProps';
import metrics from '../../../metrics';
import {classNames} from '../../../utils';

const b = classNames('registration__qr');
const sendMetric = (msg, goal) => {
    metrics.send(['QR на регистрации', msg]);
    metrics.goal(goal);
};

class Qr extends React.PureComponent {
    componentDidMount() {
        if (window.addEventListener) {
            window.addEventListener('message', this._receiveBackQrMessage, false);
        } else {
            window.attachEvent('onmessage', this._receiveBackQrMessage);
        }

        sendMetric('показ', 'desktop2mobile_qr_show');
    }

    componentWillUnmount() {
        if (window.addEventListener) {
            window.removeEventListener('message', this._receiveBackQrMessage, false);
        } else {
            window.detachEvent('onmessage', this._receiveBackQrMessage);
        }
    }

    // eslint-disable-next-line consistent-return
    _receiveBackQrMessage = (event) => {
        const {origin, data = {}} = event;
        const {magicQrUrl, retpath} = this.props;

        if (magicQrUrl !== origin) {
            return null;
        }

        if (data.backQrTokenIssued && retpath) {
            sendMetric('успешное сканирование', 'desktop2mobile_qr_sucess');
            location.href = retpath;
        }
    };

    _onBackButtonClick = () => {
        sendMetric('клик по "Вернуться на сервис"', 'desktop2mobile_qr_go_back');
    };

    _maybeRenderBackButton() {
        const {retpath} = this.props;

        if (!retpath) {
            return null;
        }

        return (
            <div className={b('back')}>
                <Button size='l' view='action' width='max' type='link' onClick={this._onBackButtonClick} url={retpath}>
                    {i18n('Restore.restore.suggest-btn')}
                </Button>
            </div>
        );
    }

    render() {
        const {magicQrUrl} = this.props;
        const iconHTML = `<span class="${b('qr-icon')}" aria-hidden="true"></span>`;
        const subtitle = i18n('OneDomik.magic.qr_icon_text').replace('%icon', iconHTML);

        return (
            <div className={b()}>
                <div className={b('title')}>{i18n('Frontend.registration_qr-title')}</div>

                <div className={b('subtitle')} dangerouslySetInnerHTML={{__html: subtitle}} />

                <iframe
                    scrolling='no'
                    name='back-qr'
                    className={b('frame')}
                    src={`${magicQrUrl}?mode=registration-qr`}
                />

                {this._maybeRenderBackButton()}
            </div>
        );
    }
}

Qr.propTypes = {
    retpath: PropTypes.string,
    isMobile: PropTypes.bool.isRequired,
    magicQrUrl: PropTypes.string.isRequired
};

export default connect(mapStateToProps)(Qr);
