import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import FormSubmit from '@blocks/registration/fieldComponents/FormSubmit.jsx';
import RegistrationFields from './components/RegistrationFields.jsx';
import validateForm from './methods/validateForm';
import EmailConfirmControls from './components/EmailConfirmControls.jsx';
import EulaBlock from './components/EulaBlock.jsx';
import {updateFormHeaderInfo} from '@blocks/registration-lite/actions';
import classnames from 'classnames';
import Footer from '@components/Footer/Footer.jsx';
import SpinnerComponent from '@blocks/registration/spinner/spinner.jsx';
import {setCurrentStep} from '@blocks/registration/actions';

export class RegistrationLite extends Component {
    static mapStateToProps(state) {
        const {
            settings: {uatraits},
            fetchingStatus: {isFetching},
            form,
            isEulaConfirmPopupShow,
            registrationType = 'default',
            experimentVersion,
            registrationStep,
            registrationErrors
        } = state;
        const isMobile = uatraits.isMobile && !uatraits.isTablet;

        return {
            isMobile,
            isFetching,
            registrationType,
            registrationStep,
            form,
            experimentVersion,
            isEulaConfirmPopupShow,
            registrationErrors
        };
    }

    constructor(props) {
        super(props);

        this.onFormSubmit = this.onFormSubmit.bind(this);
        this.renderEmailConfirmation = this.renderEmailConfirmation.bind(this);
        this.updatePasswordStepStatus = this.updatePasswordStepStatus.bind(this);
        this.renderForm = this.renderForm.bind(this);
    }

    onFormSubmit(event) {
        event.preventDefault();
        this.props.dispatch(validateForm());
    }

    renderEmailConfirmation() {
        const {form, isMobile, registrationErrors} = this.props;
        const {values, states, errors} = form;
        const isConfirmationCodeSent = states.emailCodeStatus === 'code_sent';
        const isCheckCodeLimitExceededError =
            errors.emailCode && errors.emailCode.code === 'email_confirmations_limit.exceeded';

        return (
            <EmailConfirmControls
                registrationErrors={registrationErrors}
                dispatch={this.props.dispatch}
                emailValue={values.email}
                emailCodeValue={values.emailCode}
                emailCodeStatus={states.emailCode}
                isCodeSent={isConfirmationCodeSent}
                eulaChecked={true}
                isEulaShowedInPopup={false}
                isMobile={isMobile}
                isCheckCodeLimitExceededError={isCheckCodeLimitExceededError}
            />
        );
    }

    updatePasswordStepStatus() {
        const {dispatch} = this.props;

        dispatch(
            updateFormHeaderInfo({
                title: i18n('_AUTH_.field_password')
            })
        );
        setTimeout(() => {
            // PASSP-23883
            dispatch(setCurrentStep('password'));
        }, 0);
    }

    renderForm() {
        const isSubmitBtnDisabled = false;
        const {registrationStep, isMobile, dispatch} = this.props;
        const isSubmitRendered = registrationStep === 'password';
        const isEulaConfirmPopupShow = registrationStep === 'eula';

        if (isEulaConfirmPopupShow) {
            return <EulaBlock isMobile={isMobile} dispatch={dispatch} />;
        }

        if (registrationStep === 'letter') {
            return <div className='lite-experiment'>{this.renderEmailConfirmation()}</div>;
        }
        return (
            <div className='lite-experiment'>
                <form className={classnames('registration__form', {registration__form_desktop: !isMobile})}>
                    <RegistrationFields
                        isMobile={isMobile}
                        isPasswordStepVisible={registrationStep === 'password'}
                        showPasswordFields={this.updatePasswordStepStatus}
                    />

                    {isSubmitRendered && (
                        <FormSubmit
                            onSubmit={this.onFormSubmit}
                            size={isMobile ? 'l' : 'm'}
                            isLoading={this.isLoading}
                            disable={isSubmitBtnDisabled}
                        />
                    )}
                </form>
            </div>
        );
    }

    render() {
        const {isMobile, isFetching} = this.props;

        if (isMobile) {
            return (
                <div className='lite-experiment-mobile'>
                    {this.renderForm()}
                    <Footer />
                    <div className={classnames('fade', {'fade_is-visible': isFetching})}>
                        <SpinnerComponent />
                    </div>
                </div>
            );
        }

        return <div>{this.renderForm()}</div>;
    }
}

export default connect(RegistrationLite.mapStateToProps)(RegistrationLite);

RegistrationLite.propTypes = {
    registrationType: PropTypes.string,
    registrationStep: PropTypes.string.isRequired,
    form: PropTypes.object.isRequired,
    isEulaConfirmPopupShow: PropTypes.bool,
    dispatch: PropTypes.func.isRequired,
    isMobile: PropTypes.bool.isRequired,
    isFetching: PropTypes.bool.isRequired,
    registrationErrors: PropTypes.object
};
