import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {Input} from '@components/Input';
import {Link} from '@components/Link';
import FieldPopup from '@blocks/registration/desktop/errorPopup/error_popup.jsx';
import {updateValues, updateErrorsValid} from '@blocks/actions/form';
import classnames from 'classnames';

class EmailConfirmCodeInput extends PureComponent {
    static mapStateToProps(state) {
        const {form, settings} = state;
        const status = form.states.emailCode;
        const error = form.errors.emailCode;
        const activeError = form.errors.active;
        const value = form.values.emailCode;
        const isMobile = Boolean(
            (settings.uatraits.isMobile || settings.uatraits.isTouch) && !settings.uatraits.isTablet
        );

        return {
            error,
            activeError,
            value,
            status,
            isMobile
        };
    }

    constructor(props) {
        super(props);

        this.handleInput = this.handleInput.bind(this);
    }

    handleInput(event) {
        const {target = {}} = event;
        const {value} = target;
        const fieldInfo = {
            field: 'emailCode',
            value
        };

        this.props.dispatch(updateValues(fieldInfo));
        this.props.dispatch(updateErrorsValid('emailCode'));
    }

    renderError() {
        const {error, activeError, isMobile} = this.props;
        const notValid = status === 'not_valid';

        if (isMobile) {
            return (
                <div className='form__field-error error-message'>
                    <div className='field__error-message-text'>{error.text}</div>
                </div>
            );
        }

        return (
            <FieldPopup fieldName='emailCode' visible={notValid && activeError === 'emailCode'}>
                <div className='form__popup-error' data-t='emailCode-error'>
                    <div className='error-message'>{error.text}</div>
                </div>
            </FieldPopup>
        );
    }

    render() {
        const {status, value, sendEmailWithCode, isCodeUpdated} = this.props;
        const notValid = status === 'not_valid';
        const blockCssClasses = classnames('form__field form__field_wide email__confirm_code-input', {
            form__field_filled: value,
            field__error: notValid,
            field__valid: status === 'valid'
        });

        return (
            <div className={blockCssClasses}>
                <Input
                    id='email-code'
                    name='email-code'
                    autoFocus={true}
                    value={value}
                    onChange={this.handleInput}
                    size='l'
                />
                <label htmlFor='email-code' className='registration__label'>
                    {i18n('Profile.emails.page.label.code')}
                </label>
                {this.renderError()}
                <div className='registration__pseudo-link link_send_email'>
                    <Link onClick={sendEmailWithCode}>{i18n('_AUTH_.phone_send_code_again')}</Link>
                </div>
                <span className={classnames('code-sent_note', {'show-note': isCodeUpdated})}>
                    {i18n('_AUTH_.code_new_send_message')}
                </span>
            </div>
        );
    }
}

export default connect(EmailConfirmCodeInput.mapStateToProps)(EmailConfirmCodeInput);

EmailConfirmCodeInput.propTypes = {
    value: PropTypes.string.isRequired,
    status: PropTypes.string.isRequired,
    error: PropTypes.object.isRequired,
    activeError: PropTypes.string.isRequired,
    sendEmailWithCode: PropTypes.func.isRequired,
    isCodeUpdated: PropTypes.bool,
    isMobile: PropTypes.bool,
    dispatch: PropTypes.func.isRequired
};
