import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import EmailConfirmCodeInput from './EmailConfirmCodeInput.jsx';
import SpinnerComponent from '@blocks/registration/spinner/spinner.jsx';
import registrationLiteMethods from '@blocks/registration-lite/methods/registrationLiteMethods';
import EulaPopup from '@blocks/registration/desktop/eula/EulaPopup.jsx';
import {updateStates, updateValues, updateErrorsValid} from '@blocks/actions/form';
import classnames from 'classnames';
import metrics from '@blocks/metrics';
import sendConfirmationCode from '@blocks/registration-lite/methods/sendConfirmationCode';

export default class EmailConfirmControls extends Component {
    constructor(props) {
        super(props);

        this.sendEmailWithCode = this.sendEmailWithCode.bind(this);
        this.checkEmailCode = this.checkEmailCode.bind(this);
        this.sendFormData = this.sendFormData.bind(this);
        this.handleUpdatedState = this.handleUpdatedState.bind(this);

        this.state = {
            emailCodeUpdated: false
        };
    }

    sendEmailWithCode() {
        const {dispatch} = this.props;

        dispatch(updateErrorsValid('emailCode'));
        dispatch(updateValues({field: 'emailCode', value: ''}));
        dispatch(updateStates({field: 'emailCode', status: ''}));
        dispatch(sendConfirmationCode());
        this.setState({
            emailCodeUpdated: true
        });
        this.handleUpdatedState();
    }

    handleUpdatedState() {
        setTimeout(() => {
            this.setState({
                emailCodeUpdated: false
            });
        }, 2000);
    }

    checkEmailCode() {
        const {dispatch, emailCodeValue} = this.props;

        dispatch(registrationLiteMethods.checkEmailCode(emailCodeValue));
        metrics.send(['Форма', 'Ввод кода из письма']);
    }

    sendFormData() {
        this.props.dispatch(registrationLiteMethods.sendRegistrationLiteData());
    }

    render() {
        const {
            isCodeSent,
            emailValue,
            isCheckCodeLimitExceededError,
            isFetching,
            registrationErrors,
            isMobile,
            isEulaShowedInPopup
        } = this.props;
        const codeSentTxt = i18n('_AUTH_.reg.lite_send-code')
            .replace('%', emailValue)
            .replace('<strong>', '')
            .replace('</strong>', '');
        const isRegErrorOccured = registrationErrors && registrationErrors.status === 'error';
        const isRegBtnDisabled = isCheckCodeLimitExceededError || isRegErrorOccured;

        return (
            <div
                className={classnames('registration__form block_code-sent show_code-sent', {
                    registration__form_desktop: !isMobile
                })}
            >
                <div className='email__confirm-controls'>
                    <div className='email__confirm_send-code'>
                        <div className='email__confirm_send-text registration__form-text'>{codeSentTxt}</div>
                        <EmailConfirmCodeInput
                            isCodeUpdated={this.state.emailCodeUpdated && isCodeSent}
                            sendEmailWithCode={this.sendEmailWithCode}
                            updateUserField={this.props.updateUserField}
                        />
                        {isFetching && <SpinnerComponent />}
                        <div className='lite-reg_submit'>
                            <Button
                                view='action'
                                disabled={isRegBtnDisabled}
                                size='l'
                                width='max'
                                onClick={this.checkEmailCode}
                            >
                                {i18n('Frontend.reg.lite_finish-registration')}
                            </Button>
                            {isEulaShowedInPopup && <EulaPopup sendData={this.sendFormData} />}
                        </div>
                        {isMobile && registrationErrors.status === 'error' && (
                            <div className='error-message'>
                                <strong>{registrationErrors.text}</strong>
                                {registrationErrors.descriptionText && (
                                    <p dangerouslySetInnerHTML={{__html: registrationErrors.descriptionText}} />
                                )}
                            </div>
                        )}
                    </div>
                </div>
            </div>
        );
    }
}

EmailConfirmControls.propTypes = {
    dispatch: PropTypes.func.isRequired,
    emailValue: PropTypes.string,
    isCodeSent: PropTypes.bool,
    emailCodeValue: PropTypes.string,
    updateUserField: PropTypes.func,
    isCheckCodeLimitExceededError: PropTypes.bool,
    isFetching: PropTypes.bool,
    registrationErrors: PropTypes.object,
    isMobile: PropTypes.bool,
    isEulaShowedInPopup: PropTypes.bool
};
