import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {Input} from '@components/Input';
import {Link} from '@components/Link';
import classnames from 'classnames';
import FieldPopup from '../../registration/desktop/errorPopup/error_popup.jsx';
import checkIfInvalid from '../../registration/methods/checkIfInvalid';
import registrationLiteMethods from '../methods/registrationLiteMethods';
import debounce from 'lodash/debounce';
import metrics from '../../metrics';

class RegistrationEmailInput extends PureComponent {
    static mapStateToProps(state) {
        const {
            form,
            common: {query}
        } = state;
        const status = form.states.email;
        const error = form.errors.email;
        const activeError = form.errors.active;
        const value = form.values.email;

        return {
            error,
            activeError,
            value,
            status,
            query
        };
    }

    constructor(props) {
        super(props);

        this.state = {
            isActive: false
        };
        this.handleInput = this.handleInput.bind(this);
        this.checkInvalid = this.checkInvalid.bind(this);
        this.handleBlur = this.handleBlur.bind(this);
        this.handleValidation = debounce(this.handleValidation.bind(this), 600);
        this.addMainRegistrationLink = this.addMainRegistrationLink.bind(this);
        this.metricsLogClick = this.metricsLogClick.bind(this);
    }

    componentDidMount() {
        const {value} = this.props;

        if (value.trim()) {
            const length = value.length;

            this.email.setSelectionRange(length, length);
        }
    }

    metricsLogClick() {
        metrics.send(['Лайтовая регистрация', 'Переход по ссылке "Завести новую почту на Яндексе"']);
    }

    handleInput(event) {
        const {target = {}} = event;
        const {value} = target;
        const {updateUserField} = this.props;

        updateUserField(value, {name: 'email'});
        this.handleValidation(value);
    }

    handleValidation(value) {
        this.props.dispatch(registrationLiteMethods.validateEmail(value));
    }

    checkInvalid() {
        const {value} = this.props;

        this.setState({isActive: true});
        this.props.dispatch(checkIfInvalid('email'));

        if (value.trim()) {
            this.handleValidation(value);
        }
    }

    handleBlur() {
        const metricsData = ['Ввод email'];

        this.setState({isActive: false});
        metrics.send(metricsData);
    }

    addMainRegistrationLink() {
        const {value, query, status} = this.props;
        const isValid = status === 'valid';
        const isFieldActive = this.state.isActive;

        let url = '/registration';
        const filteredQuery = Object.keys(query).filter((key) => key !== 'login');

        if (filteredQuery.length > 0) {
            url = `/registration?${filteredQuery.map((key) => `${key}=${query[key]}`).join('&')}&skipLite=1`;
        }

        return (
            <div
                className={classnames('registration__pseudo-link link_send_email', {
                    is_hidden: isValid && value && !isFieldActive
                })}
            >
                <Link href={url} onClick={this.metricsLogClick} tabIndex='0'>
                    {i18n('Frontend.reg.lite_create_account')}
                </Link>
            </div>
        );
    }

    render() {
        const {status, value, activeError, error, isAutoFocuseDisabled} = this.props;
        const notValid = status === 'not_valid';
        const isFieldActive = this.state.isActive;
        const blockCssClasses = classnames('form__field form__field_wide', {
            form__field_filled: value,
            field__error: notValid,
            field__valid: status === 'valid'
        });

        return (
            <div className={blockCssClasses}>
                <Input
                    size='l'
                    id='email'
                    name='email'
                    value={value}
                    autoFocus={!isAutoFocuseDisabled}
                    controlRef={(email) => (this.email = email)}
                    onBlur={this.handleBlur}
                    onFocus={this.checkInvalid}
                    onChange={this.handleInput}
                />
                <label htmlFor='email' className='registration__label'>
                    {i18n('Frontend.field_email_label')}
                </label>
                {this.addMainRegistrationLink()}
                <FieldPopup fieldName='email' visible={notValid && activeError === 'email'}>
                    <div className='form__popup-error' data-t='email-error'>
                        <div className='error-message'>{error.text}</div>
                        {isFieldActive && <span className='suggest__description-text'>{error.descriptionText}</span>}
                    </div>
                </FieldPopup>
            </div>
        );
    }
}

export default connect(RegistrationEmailInput.mapStateToProps)(RegistrationEmailInput);

RegistrationEmailInput.propTypes = {
    value: PropTypes.string.isRequired,
    status: PropTypes.string.isRequired,
    error: PropTypes.object.isRequired,
    activeError: PropTypes.string.isRequired,
    query: PropTypes.object,
    updateUserField: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    isAutoFocuseDisabled: PropTypes.bool
};
