import api from '../../api';
import updateFieldStatus from '../../registration/methods/updateFieldStatus';
import findFieldsWithErrors from '../../registration/methods/findFieldsWithErrors';
import validatePassword from '../../registration/methods/validatePassword';
import {getFormValues} from '../../registration/methods/validateForm';
import {moveToPrevStep} from './validateForm';
import {updateRegistrationErrors, updateFetchingStatus, setCurrentStep} from '../../registration/actions';
import {updateEmailConfirmationStatus, updateFormHeaderInfo} from '../actions';
import metrics from '../../metrics';
import {errorsTxt} from '../errors';
import {errorsTxt as commonErrorsTxt} from '../../registration/errors';

const globalRegistrationError = {
    status: 'global',
    code: 'global',
    text: errorsTxt.global.errors.global
};

const registrationLiteMethods = {
    checkEmailCode,
    prepareLiteFormData,
    sendRegistrationLiteData,
    validateEmail
};

function checkEmailCode(code) {
    return (dispatch, getState) => {
        const state = getState();
        const data = {
            key: code ? code.trim() : '',
            csrf_token: state.common.csrf,
            track_id: state.common.track_id
        };

        dispatch(updateFetchingStatus(true));
        api.request('registration-code.confirm', data)
            .then((response) => {
                if (response && response.status === 'ok') {
                    updateFieldStatus('emailCode', 'valid')(dispatch, getState);
                    dispatch(setCurrentStep('eula'));
                    dispatch(
                        updateFormHeaderInfo({
                            title: i18n('_AUTH_.acceptance.popup.title')
                        })
                    );
                    dispatch(updateEmailConfirmationStatus(true));
                    metrics.send(['Форма', 'Успешная проверка кода из письма']);
                } else {
                    dispatch(updateRegistrationErrors(globalRegistrationError));
                }
            })
            .fail((error) => {
                const errorTexts = errorsTxt.emailCode.errors;
                const errors = error.error;

                if (Array.isArray(errors)) {
                    const errorObj = {
                        code: errors[0],
                        text: errors.map((code) => errorTexts[code])
                    };

                    if (errors[0] === 'track.invalid_state') {
                        // track is outdated, the process should be restarted
                        window.location.href = state.settings.location;
                        return;
                    }
                    if (errors[0] === 'email.already_confirmed') {
                        dispatch(updateEmailConfirmationStatus(true));
                        dispatch(
                            updateFormHeaderInfo({
                                title: i18n('_AUTH_.acceptance.popup.title')
                            })
                        );
                        dispatch(setCurrentStep('eula'));
                        return;
                    }
                    updateFieldStatus('emailCode', 'not_valid')(dispatch, errorObj);
                    metrics.send(['Форма', `Проверка кода из письма, ошибка: ${errorObj.code}`]);
                } else {
                    dispatch(updateRegistrationErrors(globalRegistrationError));
                }
            })
            .always(() => {
                dispatch(updateFetchingStatus(false));
            });
    };
}

function prepareLiteFormData(fieldsStatesObj, validationMethod, isMobile = false) {
    const preparedFieldsStates = Object.assign({}, fieldsStatesObj);
    const phoneConfirmed =
        preparedFieldsStates.phoneCodeStatus === 'code_sent' && preparedFieldsStates.phoneCode === 'valid';

    delete preparedFieldsStates.login;
    delete preparedFieldsStates.hint_question_id; // it is always has some value so it is always valid
    delete preparedFieldsStates.phoneCode;
    delete preparedFieldsStates.emailCode;
    delete preparedFieldsStates.emailCodeStatus;

    if (phoneConfirmed || validationMethod === 'captcha') {
        delete preparedFieldsStates.phoneCodeStatus;
    }

    if (validationMethod === 'phone') {
        delete preparedFieldsStates.hint_question_id;
        delete preparedFieldsStates.hint_question_custom;
        delete preparedFieldsStates.hint_question;
        delete preparedFieldsStates.hint_answer;
        delete preparedFieldsStates.captcha;
    }

    if (validationMethod === 'captcha') {
        delete preparedFieldsStates.phone;
    }

    if (isMobile) {
        delete preparedFieldsStates.password_confirm;
    }

    return preparedFieldsStates;
}

function sendRegistrationLiteData() {
    return (dispatch, getState) => {
        const state = getState();
        const trackId = state.common.track_id;
        const baseRedirectUrl = `/registration/finish?track_id=${trackId}`;
        const method = state.form.validation.method;
        const params = Object.assign({}, getFormValues(state), {'human-confirmation': method});

        dispatch(updateFetchingStatus(true));
        params.csrf_token = state.common.csrf;
        params.login = params.email;
        params.track_id = trackId;
        params.eula_accepted = 'on';

        api.request('registration-lite', params)
            .then((response) => {
                if (response.status === 'ok') {
                    metrics.send(['Форма', 'Успешная lite-регистрация']);
                    window.location.href = baseRedirectUrl;
                } else {
                    dispatch(updateRegistrationErrors(globalRegistrationError));
                }
            })
            .fail((error) => {
                handleRegistrationError(error, dispatch, state.common.currentPage);
            });
    };
}

function validateEmail(email) {
    return (dispatch, getState) => {
        const state = getState();
        const trackId = state.common.track_id;
        const errors = errorsTxt.email.errors;
        const errorDescriptions = errorsTxt.email.descriptions;
        const password = state.form.values.password;
        const errorObj = {
            code: 'global',
            text: errorsTxt.global.errors.global,
            descriptionText: ''
        };

        api.request('login', {track_id: trackId, login: email, is_lite: 1}).then((response) => {
            if (response !== undefined) {
                const errorCode = Array.isArray(response) ? response : null;

                if (errorCode) {
                    errorObj.code = errorCode[0];
                    errorObj.text = errorCode.map((code) => errors[code])[0];
                    errorObj.descriptionText = errorCode.map((code) => errorDescriptions[code])[0];
                }

                if (response.status === 'ok') {
                    updateFieldStatus('email', 'valid')(dispatch, getState);
                    dispatch(findFieldsWithErrors());
                    metrics.send(['Форма', 'Успешное заполнение поля "email"']);
                } else {
                    updateFieldStatus('email', 'not_valid')(dispatch, errorObj);
                    metrics.send(['Форма', `Заполнение поля "email", ошибка: ${errorObj.code}`]);
                }
            }
        });

        if (password) {
            dispatch(validatePassword(password, 'email'));
        }
    };
}

export default registrationLiteMethods;

export function handleRegistrationError(error, dispatch, currentPageUrl) {
    if (error && Array.isArray(error.error)) {
        const field = returnErrorField(error.error[0]);
        const errorCode = returnErrorCode(error.error[0]);

        moveToPrevStep([field], dispatch);
        if (errorCode === 'registrationsmssendperiplimitexceeded') {
            const regError = {
                status: 'error',
                code: 'registrationsmssendperiplimitexceeded',
                text: commonErrorsTxt.registration.errors.registrationsmssendperiplimitexceeded
            };

            return dispatch(updateRegistrationErrors(regError));
        }
        const errorTxt =
            field && commonErrorsTxt[field].errors ? commonErrorsTxt[field].errors[errorCode] : commonErrorsTxt.global;

        if (errorCode === 'sessionidoverflow') {
            const regError = {
                status: 'error',
                code: 'sessionidoverflow',
                text: commonErrorsTxt.registration.errors.sessionidoverflow,
                descriptionText: commonErrorsTxt.registration.descriptions.sessionidoverflow
            };

            return dispatch(updateRegistrationErrors(regError));
        }
        if (errorCode === 'registrationalreadycompletederror') {
            // PASSP-17624
            window.location.href = currentPageUrl;
        }
        updateFieldStatus(field, 'not_valid')(dispatch, {code: errorCode, text: errorTxt});
        metrics.send(['Форма', 'Показ ошибок', errorCode]);
        return dispatch(updateFetchingStatus(false));
    }
    return dispatch(updateRegistrationErrors(globalRegistrationError));
}

function returnErrorField(errorObj) {
    if (errorObj.field) {
        return errorObj.field;
    }
    return errorObj.split('.')[0];
}

function returnErrorCode(errorObj) {
    if (errorObj.code) {
        return errorObj.code;
    }
    return errorObj.split('.')[1];
}
