import api from '@blocks/api';
import {
    setCurrentStep,
    showEulaPopup,
    updateFetchingStatus,
    updateRegistrationErrors
} from '@blocks/registration/actions';
import {updateStates} from '@blocks/actions/form';
import {updateEmailConfirmationStatus, updateFormHeaderInfo} from '@blocks/registration-lite/actions';
import metrics from '@blocks/metrics';
import {errorsTxt} from '@blocks/registration-lite/errors';
import updateFieldStatus from '@blocks/registration/methods/updateFieldStatus';

const globalRegistrationError = {
    status: 'error',
    code: 'global',
    text: errorsTxt.global.errors.global
};

export default function() {
    return (dispatch, getState) => {
        const state = getState();
        const {form, settings, common} = state;
        const {values} = form;
        const {language} = settings;
        const formData = {
            track_id: common.track_id,
            language,
            login: values.email
        };

        return api
            .request('registration-lite-experiment.submit', formData)
            .then((response) => {
                if (response && response.status === 'ok') {
                    dispatch(updateStates({field: 'emailCodeStatus', status: 'code_sent'}));
                    dispatch(setCurrentStep('letter'));
                    dispatch(
                        updateFormHeaderInfo({
                            title: i18n('Frontend.reg.lite_email-confirm')
                        })
                    );
                    metrics.send(['Форма', 'Успешная отправка письма с кодом']);
                } else {
                    dispatch(updateRegistrationErrors(globalRegistrationError));
                }
            })
            .fail((error) => {
                const errorTexts = errorsTxt.email.errors;
                const errors = error.error;

                if (Array.isArray(errors)) {
                    const errorCode = error.error[0];
                    const errorObj = {
                        code: errors[0],
                        text: errors.map((code) => errorTexts[code])[0] || errorsTxt.global.errors.global
                    };

                    if (errorCode === 'email.already_confirmed') {
                        dispatch(updateEmailConfirmationStatus(true));
                        dispatch(showEulaPopup(true));
                        dispatch(
                            updateFormHeaderInfo({
                                title: i18n('_AUTH_.acceptance.popup.title')
                            })
                        );
                        dispatch(setCurrentStep('eula'));
                        return;
                    }
                    updateFieldStatus('email', 'not_valid')(dispatch, errorObj);
                    metrics.send(['Форма', `Ошибка отправки письма с кодом, ${errorObj.code}`]);
                } else {
                    dispatch(updateRegistrationErrors(globalRegistrationError));
                }
            })
            .always(() => {
                dispatch(updateFetchingStatus(false));
            });
    };
}
