import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {setCurrentStep} from '@blocks/registration/actions';
import {updateStates, updateValues, updateErrorsValid} from '@blocks/actions/form';
import {updateEmailConfirmationStatus, updateFormHeaderInfo} from '@blocks/registration-lite/actions';
import classnames from 'classnames';
import {PlusPromoSlogan} from '@blocks/registration/customizations/PlusPromo/PlusPromoSlogan';
import Logo from '@blocks/registration/customizations/Logo';

export class FormHeader extends Component {
    static mapStateToProps(state) {
        const {
            settings: {language, uatraits, staticPath},
            registrationType,
            isIconsExperiment,
            experimentsList = {},
            registrationStep,
            registrationErrors: {status: errorStatus}
        } = state;
        const titleText = i18n('Frontend.registration_main-title_id');

        return {
            uatraits,
            language,
            registrationType,
            isIconsExperiment,
            titleText,
            registrationStep,
            errorStatus,
            isPlusPromo: experimentsList.isPlusPromo,
            staticPath
        };
    }

    constructor(props) {
        super(props);

        this.goStepBack = this.goStepBack.bind(this);
        this.updateHeaderTitle = this.updateHeaderTitle.bind(this);
    }

    goStepBack() {
        const {registrationStep, dispatch} = this.props;
        const stepMaps = {
            password: 'start',
            letter: 'password',
            eula: 'letter'
        };

        dispatch(setCurrentStep(stepMaps[registrationStep]));
        this.updateHeaderTitle(stepMaps[registrationStep]);

        if (registrationStep === 'eula') {
            dispatch(updateStates({field: 'emailCode', status: ''}));
            dispatch(setCurrentStep('letter'));
            dispatch(updateEmailConfirmationStatus(false));
        }

        if (registrationStep === 'letter') {
            dispatch(updateStates({field: 'emailCodeStatus', status: ''}));
            dispatch(updateStates({field: 'emailCode', status: ''}));
            dispatch(updateValues({field: 'emailCode', value: ''}));
            dispatch(updateErrorsValid('emailCode'));
        }
    }

    updateHeaderTitle(step) {
        const titleMaps = {
            start: i18n('Frontend.registration_main-title'),
            password: i18n('_AUTH_.field_password'),
            letter: i18n('Frontend.reg.lite_email-confirm'),
            eula: i18n('_AUTH_.acceptance.popup.title')
        };

        this.props.dispatch(updateFormHeaderInfo({title: titleMaps[step]}));
    }

    render() {
        const {
            uatraits,
            isIconsExperiment,
            registrationStep,
            errorStatus,
            registrationType,
            titleText,
            isPlusPromo,
            staticPath
        } = this.props;
        const isMobile = uatraits.isMobile && !uatraits.isTablet;
        const isPrevButtonShown = registrationStep && registrationStep !== 'start';

        if (errorStatus === 'error') {
            return null;
        }

        if (isMobile && registrationType === 'childish') {
            return (
                <div className='registration-mobile__header'>
                    <h1 className='registration-mobile__title'>{i18n('_AUTH_.registration-title.childish')}</h1>
                </div>
            );
        }

        if (isMobile && !isIconsExperiment) {
            return (
                <div className='registration-mobile__header'>
                    {isPrevButtonShown && (
                        <button
                            type='button'
                            className='reg-form_back-btn'
                            title={i18n('_AUTH_.common.back')}
                            onClick={this.goStepBack}
                        />
                    )}
                    <Logo origin='default' staticPath={staticPath} />

                    <h1 className='registration-mobile__title'>{titleText}</h1>
                    {isPlusPromo && <PlusPromoSlogan />}
                </div>
            );
        }

        if (registrationType === 'lite') {
            return (
                <div
                    className={classnames('registration-desktop__header', {
                        step__header: registrationStep !== undefined
                    })}
                >
                    {isPrevButtonShown && (
                        <button
                            type='button'
                            className='reg-form_back-btn'
                            title={i18n('_AUTH_.common.back')}
                            onClick={this.goStepBack}
                        />
                    )}
                    <h1 className='header__descr'>{titleText}</h1>
                </div>
            );
        }

        return null;
    }
}

export default connect(FormHeader.mapStateToProps)(FormHeader);

FormHeader.propTypes = {
    uatraits: PropTypes.object.isRequired,
    language: PropTypes.string.isRequired,
    titleText: PropTypes.string,
    page: PropTypes.string,
    registrationStep: PropTypes.string,
    registrationType: PropTypes.string,
    isIconsExperiment: PropTypes.bool,
    dispatch: PropTypes.func.isRequired,
    errorStatus: PropTypes.string,
    isPlusPromo: PropTypes.bool,
    staticPath: PropTypes.string.isRequired
};
