import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';
import {hasExp} from '@blocks/selectors';
import FormSubmit from '../fieldComponents/FormSubmit.jsx';
import AuthLink from '../fieldComponents/AuthLink.jsx';
import MobileFooter from '../../registration/mobile/MobileFooter.jsx';
import HumanConfirmation from '../fieldComponents/human_confirmation/human_confirmation.jsx';
import FormErrors from './form_errors/form_errors.jsx';
import Eula from './eula/Eula.jsx';
import RegistrationTitle from '../fieldComponents/RegistrationTitle.jsx';
import UserName from '../fieldComponents/username.jsx';
import UserPassword from '../fieldComponents/userpassword.jsx';
import {RadioButton} from '@components/RadioButton';
import {ChildishRegLanding} from '../childishRegLanding';
// import SocialButtons from '../socialButtons/socialButtons.jsx';
import regMethods from '../methods/basicRegistrationMethods';
import returnFormData from '../methods/returnFormData';
import {REGISTRATION_GOAL_PREFIX} from '../actions';
import validateForm from '../methods/validateForm';
import metrics from '../../metrics';

const VIEW_RADIO_OPTIONS = [
    {value: 'default', children: i18n('_AUTH_.reg-childish-radio.option.1'), 'data-t': 'registration:view:default'},
    {
        value: 'childishLanding',
        children: i18n('_AUTH_.reg-childish-radio.option.2'),
        'data-t': 'registration:view:childishLanding'
    }
];

export class RegistrationForm extends PureComponent {
    static mapStateToProps(state) {
        const {settings, common, form, fetchingStatus} = state;
        const {formUrl, uatraits, authUrl, origin} = settings;
        const {track_id: trackId, familyChildishRegOriginUrl} = common;
        const isMobile = uatraits.isMobile && !uatraits.isTablet;
        const {isFetching} = fetchingStatus;
        const hasChildishExp = hasExp(state, 'profile-family-childish-exp');

        return {
            formUrl,
            authUrl,
            isFetching,
            trackId,
            form,
            isMobile,
            origin,
            hasChildishExp,
            familyChildishRegOriginUrl
        };
    }

    constructor(props) {
        super(props);
        const {isEulaCheckedByDefault, isEulaShowedInPopup} = props.form;

        this.onFormSubmit = this.onFormSubmit.bind(this);
        this.handleEula = this.handleEula.bind(this);
        this.sendRegistrationData = this.sendRegistrationData.bind(this);
        this.renderMobile = this.renderMobile.bind(this);
        this.renderForm = this.renderForm.bind(this);

        this.state = {
            eulaChecked: isEulaCheckedByDefault || isEulaShowedInPopup,
            regView: 'default'
        };

        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;
        this.forceUpdate();
    }

    onFormSubmit(evt) {
        evt.preventDefault();
        metrics.send(['Форма', 'Попытка отправки']);
        metrics.goal(`${REGISTRATION_GOAL_PREFIX}_form_submitted`);

        this.props.dispatch(validateForm());
    }

    sendRegistrationData() {
        const {dispatch} = this.props;
        const formData = dispatch(returnFormData());

        dispatch(regMethods.submitRegistration(formData));
        metrics.send(['Форма', 'Отправка формы']);
        metrics.goal(`${REGISTRATION_GOAL_PREFIX}_form_valid`);
    }

    handleEula() {
        this.setState((prevState) => ({
            eulaChecked: !prevState.eulaChecked
        }));
    }

    handleRegViewRadio = (evt = {}) => {
        const {currentTarget: {value} = {}} = evt;

        this.setState({regView: value});
    };

    renderMobile() {
        const {isFetching, isMobile} = this.props;

        return (
            <div className='experiment-mobile'>
                {this.renderForm()}
                <MobileFooter isFetching={isFetching} isMobile={isMobile} />
            </div>
        );
    }

    childishLandingHandler = () => {
        const {familyChildishRegOriginUrl} = this.props;

        window.location.href = familyChildishRegOriginUrl;
    };

    renderChildishLanding = () => {
        const {isMobile} = this.props;

        return (
            <div
                className={classnames('registration__childishLanding', {
                    registration__childishLanding_desktop: !isMobile,
                    registration__childishLanding_mobile: isMobile
                })}
            >
                <ChildishRegLanding
                    onNextButtonClick={this.childishLandingHandler}
                    showCloseButton={false}
                    showTitle={false}
                />
            </div>
        );
    };

    renderForm() {
        const {formUrl, authUrl, form, trackId, isMobile, dispatch, origin, hasChildishExp} = this.props;
        const {activeField, validation, states, values, errors, formErrors, isEulaShowedInPopup, type} = form;
        const phoneConfirmationInProcess = states.phoneCodeStatus === 'code_sent' && states.phoneCode !== 'valid';
        const {eulaChecked, regView} = this.state;
        const isSubmitBtnDisabled = phoneConfirmationInProcess || this.props.isFetching || !eulaChecked;
        const isDefaultReg = type === 'alternative';
        const isChildishReg = type === 'childish';
        const title = isChildishReg ? i18n('_AUTH_.registration-title.childish') : i18n('_AUTH_.registration-title');
        const submitBtnText = isChildishReg
            ? i18n('_AUTH_.registration.submit-btn.childish')
            : i18n('Profile.regpromo.register');

        return (
            <div>
                <RegistrationTitle isMobile={isMobile} title={title} />
                {/* @see https://st.yandex-team.ru/PASSP-29953#5f75b4893a12e9550a67c8b9 */}
                {/*<SocialButtons place='alternative' />*/}
                {hasChildishExp && isDefaultReg && (
                    <div
                        className={classnames('registration__radio', {
                            registration__radio_desktop: !isMobile,
                            registration__radio_mobile: isMobile
                        })}
                    >
                        <RadioButton
                            width='max'
                            value={regView}
                            options={VIEW_RADIO_OPTIONS}
                            onChange={this.handleRegViewRadio}
                        />
                    </div>
                )}

                {regView === 'childishLanding' && hasChildishExp && this.renderChildishLanding()}

                {regView === 'default' && (
                    <form
                        className={classnames('registration__form', {
                            registration__form_desktop: !isMobile,
                            registration__form_mobile: isMobile
                        })}
                        autoComplete='off'
                        action={formUrl}
                        method='post'
                        onSubmit={this.onFormSubmit}
                    >
                        <input name='track_id' type='hidden' value={trackId} />

                        <UserName
                            isMobile={isMobile}
                            isAutoFocuseDisabled={isMobile}
                            states={states}
                            values={values}
                            errors={errors}
                            activeField={activeField}
                            dispatch={dispatch}
                            registrationType={type}
                        />

                        <UserPassword
                            isMobile={isMobile}
                            states={states}
                            values={values}
                            errors={errors}
                            dispatch={dispatch}
                        />

                        <HumanConfirmation />

                        <FormErrors errors={formErrors} />

                        <FormSubmit
                            isEulaByPopup={isEulaShowedInPopup}
                            sendData={this.sendRegistrationData}
                            onSubmit={this.onFormSubmit}
                            disable={isSubmitBtnDisabled}
                            isLoading={this.isLoading}
                            text={submitBtnText}
                        />

                        <AuthLink isMobile={isMobile} authUrl={authUrl} />

                        {!isEulaShowedInPopup && (
                            <Eula
                                showServiceSubscriptionCheckbox={Boolean(origin)}
                                eulaChecked={eulaChecked}
                                validationMethod={validation.method}
                                handleEula={this.handleEula}
                                handleEulaMoney={this.handleEulaMoney}
                                isChildishReg={isChildishReg}
                                isEulaCheckedByDefault={form.isEulaCheckedByDefault}
                            />
                        )}
                    </form>
                )}
            </div>
        );
    }

    render() {
        const {isMobile} = this.props;

        if (isMobile) {
            return this.renderMobile();
        }

        return this.renderForm();
    }
}

RegistrationForm.propTypes = {
    dispatch: PropTypes.func.isRequired,
    form: PropTypes.object.isRequired,
    formUrl: PropTypes.string.isRequired,
    trackId: PropTypes.string.isRequired,
    isFetching: PropTypes.bool,
    authUrl: PropTypes.string,
    registrationType: PropTypes.string,
    isMobile: PropTypes.bool.isRequired,
    origin: PropTypes.string,
    hasChildishExp: PropTypes.bool,
    familyChildishRegOriginUrl: PropTypes.string
};

export default connect(RegistrationForm.mapStateToProps)(RegistrationForm);
