import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Input} from '@components/Input';
import {Captcha} from '@components/Captcha';
import ControlQuestionsPopup from './hint_question-popup.jsx';
import FormErrors from '@blocks/registration/desktop/form_errors/form_errors.jsx';
import {updateStates, updateErrors} from '@blocks/actions/form';

import classnames from 'classnames';

class ControlCaptcha extends PureComponent {
    constructor(props) {
        super(props);

        this.validateCaptchaField = this.validateCaptchaField.bind(this);
        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        if (this.props.stateError === 'need_validation') {
            this.validateCaptchaField();
        }

        this.forceUpdate();
    }

    validateCaptchaField() {
        const value = this.props.fieldCaptcha;
        const isEmpty = !value.trim();
        const status = isEmpty ? 'not_valid' : 'valid';
        const captchaErrors = {
            missingvalue: i18n('_AUTH_.answer_errors_missingvalue'),
            captchalocate: i18n('_AUTH_.answer_errors_captchalocate'),
            incorrect: i18n('_AUTH_.answer_errors_incorrect')
        };

        this.props.dispatch(updateStates({field: 'captcha', status}));
        this.props.dispatch(
            updateErrors({
                field: 'captcha',
                error: {
                    code: 'missingvalue',
                    text: isEmpty ? captchaErrors.missingvalue : ''
                },
                active: isEmpty
            })
        );
    }

    render() {
        const {
            fieldCaptcha,
            stateError,
            textError,
            activeError,
            isMobileCaptcha,
            lang,
            captcha,
            captchaErrors,
            onInputChange,
            env,
            dispatch
        } = this.props;
        const notValid = stateError === 'not_valid';

        return (
            <div className='captcha-wrapper'>
                <div
                    className={classnames('form__field', {
                        form__field_filled: fieldCaptcha || this.isLoading,
                        field__error: notValid || captchaErrors.length !== 0
                    })}
                >
                    <Input
                        id='captcha'
                        value={fieldCaptcha}
                        name='captcha'
                        onChange={onInputChange}
                        onBlur={this.validateCaptchaField}
                        view='one-border'
                        state={notValid ? 'error' : ''}
                        filled={!!fieldCaptcha || this.isLoading}
                    />
                    <label htmlFor='captcha' className='registration__label'>
                        {i18n('Frontend.field_answer')}
                    </label>

                    <ControlQuestionsPopup
                        notValid={notValid}
                        error={activeError === 'captcha'}
                        anchor={this.captchaField}
                        message={textError}
                        field='captcha'
                    />

                    {captchaErrors.length !== 0 && <FormErrors errors={captchaErrors} />}
                </div>
                <div className='form__field registration__captcha'>
                    <input name='captcha_key' type='hidden' value={captcha.key} />
                    <input name='validation_method' type='hidden' value='captcha' />

                    <Captcha
                        isMobile={isMobileCaptcha}
                        lang={lang}
                        introSound={captcha.introSound}
                        captchaSound={captcha.captchaSound}
                        imageUrl={captcha.imageUrl}
                        captchaKey={captcha.key}
                        loading={captcha.loading}
                        loadingAudio={captcha.loadingAudio}
                        playing={captcha.playing}
                        type={captcha.type}
                        dispatch={dispatch}
                        env={env}
                        visible={true}
                    />
                </div>
            </div>
        );
    }
}

export default ControlCaptcha;

ControlCaptcha.propTypes = {
    fieldCaptcha: PropTypes.string.isRequired,
    isMobileCaptcha: PropTypes.bool,
    lang: PropTypes.string.isRequired,
    captcha: PropTypes.object.isRequired,
    env: PropTypes.shape({
        type: PropTypes.string
    }),
    onInputChange: PropTypes.func.isRequired,
    stateError: PropTypes.string,
    textError: PropTypes.string,
    activeError: PropTypes.string,
    captchaErrors: PropTypes.array,
    dispatch: PropTypes.func.isRequired
};
