import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';

import PhoneConfirmation from './phone_confirm/phone_confirmation.jsx';
import UserQuestions from './hint_question/user_question.jsx';
import ControlCaptcha from './hint_question/control_captcha.jsx';
import {CaptchaModal} from './captcha_modal';

import {
    updateValidationMethod,
    updatePhoneValuesToDefaults,
    updatePhoneStatesToDefaults,
    updateStates,
    updateErrors,
    updateValues,
    changePhoneConfirmationType,
    validatePhoneForCall,
    setCallConfirmationTimer,
    updateErrorsValid
} from '@blocks/actions/form';
import metrics from '@blocks/metrics';

class HumanConfirmation extends Component {
    static mapStateToProps(state) {
        const {
            settings: {uatraits: ua, language: lang, tld, formUrl, env = {}},
            form: {
                states,
                values,
                errors,
                validation,
                captchaErrors,
                humanConfirmation,
                activeErrorField,
                captchaRequired,
                canSwitchConfirmationMethod,
                isPhoneCallConfirmationAvailable,
                isForcePhoneCallConfirmation,
                isDiscardCallConfirmation,
                isPhoneCanBePrefilled
            },
            captcha
        } = state;

        return {
            states,
            values,
            errors,
            validation,
            validationMethod: validation.method,
            humanConfirmation,
            activeErrorField,
            captchaRequired,
            lang,
            tld,
            captcha,
            canSwitchConfirmationMethod,
            isPhoneCallConfirmationAvailable,
            isForcePhoneCallConfirmation,
            isDiscardCallConfirmation,
            isPhoneCanBePrefilled,
            formUrl,
            captchaErrors,
            isMobileCaptcha: Boolean(ua.isMobile || ua.isTablet),
            isMobile: ua.isMobile && !ua.isTablet,
            env
        };
    }

    constructor(props) {
        super(props);

        this.toggleHumanConfirmation = this.toggleHumanConfirmation.bind(this);
        this.onInputChange = this.onInputChange.bind(this);
        this.clearPhoneData = this.clearPhoneData.bind(this);
        this.renderHintQuestion = this.renderHintQuestion.bind(this);

        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;
        this.forceUpdate();
    }

    clearPhoneData() {
        const {isPhoneCallConfirmationAvailable, validation, dispatch} = this.props;

        dispatch(updatePhoneValuesToDefaults());
        dispatch(updatePhoneStatesToDefaults());
        dispatch(updateErrorsValid('phoneCode'));
        dispatch(updateErrorsValid('phone'));

        if (isPhoneCallConfirmationAvailable) {
            dispatch(changePhoneConfirmationType('sms'));
            dispatch(validatePhoneForCall(false));
        }

        if (validation.callConfirmationTimer) {
            clearTimeout(validation.callConfirmationTimer);
            dispatch(setCallConfirmationTimer(null));
        }
    }

    onInputChange(event) {
        const {target = {}} = event;
        const {value, name} = target;
        const {dispatch} = this.props;
        const fieldInfo = {
            field: name,
            value
        };

        dispatch(updateValues(fieldInfo));
        dispatch(updateStates({field: name, status: ''}));
        dispatch(updateErrors({field: name, error: '', active: false}));
    }

    toggleHumanConfirmation() {
        const {dispatch, validationMethod} = this.props;

        if (validationMethod === 'phone') {
            dispatch(updateValidationMethod('captcha'));
            this.clearPhoneData();
            metrics.send(['Переключение способа валидации пользователя', 'Регистрация с КВ-КО']);
        } else {
            dispatch(updateValidationMethod('phone'));
            metrics.send(['Переключение способа валидации пользователя', 'Регистрация с номером телефона']);
        }
    }

    renderHintQuestion() {
        const {
            captcha,
            env,
            lang,
            isMobileCaptcha,
            errors,
            values,
            states,
            dispatch,
            formUrl,
            captchaErrors
        } = this.props;

        return (
            <div>
                <UserQuestions toggleConfirmationMethod={this.toggleHumanConfirmation} />
                <ControlCaptcha
                    env={env}
                    fieldCaptcha={values.captcha}
                    stateError={states.captcha}
                    textError={errors.captcha && errors.captcha.text}
                    activeError={errors.active}
                    formUrl={formUrl}
                    onInputChange={this.onInputChange}
                    isMobileCaptcha={isMobileCaptcha}
                    lang={lang}
                    captcha={captcha}
                    captchaErrors={captchaErrors}
                    dispatch={dispatch}
                />
            </div>
        );
    }

    render() {
        const {canSwitchConfirmationMethod, validationMethod, humanConfirmation: {isCaptchaRequired} = {}} = this.props;
        const isPhoneValidation = validationMethod === 'phone';

        return (
            <div
                className={classnames('human-confirmation-wrapper', {
                    'can-switch': canSwitchConfirmationMethod
                })}
            >
                {isCaptchaRequired && <CaptchaModal />}
                {isPhoneValidation && <PhoneConfirmation toggleConfirmationMethod={this.toggleHumanConfirmation} />}
                {!isPhoneValidation && this.renderHintQuestion()}
            </div>
        );
    }
}

HumanConfirmation.propTypes = {
    isMobile: PropTypes.bool,
    isPhoneCanBePrefilled: PropTypes.bool,
    values: PropTypes.object.isRequired,
    states: PropTypes.object.isRequired,
    errors: PropTypes.object.isRequired,
    activeErrorField: PropTypes.object,
    dispatch: PropTypes.func.isRequired,
    humanConfirmation: PropTypes.shape({
        controlQuestion: PropTypes.object
    }),
    validation: PropTypes.shape({
        method: PropTypes.string,
        phoneConfirmationType: PropTypes.string,
        isValidPhoneForCall: PropTypes.bool,
        callConfirmationTimer: PropTypes.func
    }),
    validationMethod: PropTypes.string,
    captchaRequired: PropTypes.bool.isRequired,
    canSwitchConfirmationMethod: PropTypes.bool.isRequired,
    isMobileCaptcha: PropTypes.bool.isRequired,
    isPhoneCallConfirmationAvailable: PropTypes.bool.isRequired,
    isForcePhoneCallConfirmation: PropTypes.bool.isRequired,
    isDiscardCallConfirmation: PropTypes.bool.isRequired,
    captcha: PropTypes.object.isRequired,
    env: PropTypes.shape({
        type: PropTypes.string
    }),
    lang: PropTypes.string.isRequired,
    tld: PropTypes.string.isRequired,
    formUrl: PropTypes.string.isRequired,
    captchaErrors: PropTypes.array
};

export default connect(HumanConfirmation.mapStateToProps)(HumanConfirmation);
