import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';
import {isAndroidAm} from '@blocks/selectors';

import {Input} from '@components/Input';
import {Spin} from '@components/Spin';

import checkIfInvalid from '@blocks/registration/methods/checkIfInvalid';
import {forceCheckedConfirmationCode, updateErrors, updateValues} from '@blocks/actions/form';
import checkPhoneCode from '@blocks/registration/methods/checkPhoneCode';

import {SendCodeHintPopup, PhoneErrorPopup} from './phone_field-popups.jsx';
import CountDown from './count_down.jsx';
import ConfirmCodeBtn from './confirm_code_btn.jsx';
import FieldDescriptionText from './field_description_text.jsx';
import ConfirmCodeBtnMobile from './ConfirmCodeBtnMobile.jsx';

export class PhoneCodeInput extends Component {
    static mapStateToProps(state) {
        const {
            settings: {uatraits},
            form,
            isPhoneConfirmationOpened,
            phoneConfirmation: {callingNumberTemplate}
        } = state;
        const {
            validation,
            states,
            errors,
            values,
            humanConfirmation,
            isPhoneCallConfirmationAvailable,
            isForcePhoneCallConfirmation,
            isDiscardCallConfirmation,
            canSwitchConfirmationMethod,
            isMobileRecallBtnRemoved,
            isTouchRecallBtnRemoved
        } = form;
        const isMobile = uatraits.isMobile && !uatraits.isTablet;

        return {
            isMobile,
            validation,
            error: errors,
            phone: values.phone,
            password: values.password,
            isFetching: humanConfirmation.isFetching,
            code: values.phoneCode,
            codeStatus: states.phoneCodeStatus,
            codeValid: states.phoneCode,
            isPhoneCallConfirmationAvailable,
            isForcePhoneCallConfirmation,
            isDiscardCallConfirmation,
            canSwitchConfirmationMethod,
            isPhoneConfirmationOpened,
            callingNumberTemplate,
            isTouchRecallButtonExperiment: isTouchRecallBtnRemoved && isMobile,
            isMobileRecallButtonExperiment: isMobileRecallBtnRemoved && isMobile,
            isCodeWithFormatExp: !isAndroidAm(state)
        };
    }

    constructor(props) {
        super(props);

        this.handleEnterPress = this.handleEnterPress.bind(this);
        this.checkInvalid = this.checkInvalid.bind(this);
        this.checkCode = this.checkCode.bind(this);
        this.handleCodeInput = this.handleCodeInput.bind(this);
    }

    componentDidMount() {
        this.phoneCode.focus();
    }

    handleEnterPress(event) {
        if (event.keyCode === 13) {
            this.checkCode();
        }
    }

    handleCodeInput(event) {
        const {target = {}} = event;
        const {value} = target;
        const fieldInfo = {
            field: 'phoneCode',
            value
        };
        const {
            validation: {isForceCheckedConfirmationCode, confirmationCodeLength},
            dispatch
        } = this.props;

        dispatch(updateValues(fieldInfo));
        dispatch(updateErrors({field: 'phoneCode', error: '', active: false}));

        const codeFormatted = value.replace(/[^0-9]+/g, '');
        const isShouldForceCheckConfirmationCode =
            confirmationCodeLength &&
            codeFormatted.length === confirmationCodeLength &&
            !isForceCheckedConfirmationCode;

        if (isShouldForceCheckConfirmationCode) {
            dispatch(forceCheckedConfirmationCode());
            dispatch(checkPhoneCode(codeFormatted));
        }
    }

    checkInvalid() {
        this.props.dispatch(checkIfInvalid('phoneCode'));
    }

    checkCode() {
        const {code, dispatch} = this.props;

        dispatch(checkPhoneCode(code));
    }

    render() {
        const {
            codeValid,
            error,
            isMobile,
            isPhoneCallConfirmation,
            isDiscardCallConfirmation,
            changeNumber,
            code,
            isPhoneCallConfirmationAvailable,
            isShowCountDown,
            isPhoneConfirmationOpened,
            isMobileRecallButtonExperiment,
            canSwitchConfirmationMethod,
            isFetching,
            showHint,
            codeSentAgain,
            hint,
            showCodeHint,
            validation: {phoneConfirmationType, humanConfirmationDone},
            confirmationRetryCountdown,
            callingNumberTemplate,
            confirmationCodeLength,
            isCodeWithFormatExp
        } = this.props;
        const field = 'phoneCode';
        const fieldCssClasses = classnames('form__field phone__code-field', {
            form__field_filled: code,
            'call-confirmation-field': isPhoneCallConfirmationAvailable,
            field__error: codeValid === 'not_valid',
            field__disabled: codeValid,
            field__valid: codeValid === 'valid'
        });
        const isErrorPopupShow = !showHint || isMobile;
        const phoneCodeError =
            error.active === 'phoneCode' && error.phoneCode.hasOwnProperty('text') ? error.phoneCode.text : '';

        let phoneEnterCodeLabel = isPhoneCallConfirmation
            ? i18n('Frontend.enter_confirmation_code_after_call')
            : i18n('Frontend.phone_enter_code');

        if (isDiscardCallConfirmation && phoneConfirmationType === 'call') {
            phoneEnterCodeLabel = i18n('Frontend.discard_call_confirmation.phone_code_label');
        }

        return (
            <div>
                {codeValid !== 'valid' && (
                    <FieldDescriptionText
                        isMobile={isMobile}
                        callingNumberTemplate={callingNumberTemplate}
                        isPhoneCallConfirmation={isPhoneCallConfirmation}
                        isDiscardCallConfirmation={isDiscardCallConfirmation}
                        changeText={codeSentAgain}
                        changeNumber={changeNumber}
                    />
                )}

                <div className={fieldCssClasses}>
                    {isShowCountDown && !isMobile && (
                        <CountDown
                            confirmationRetryCountdown={confirmationRetryCountdown}
                            restartTrigger={isPhoneCallConfirmation}
                            showCodeHint={showCodeHint}
                        />
                    )}

                    <div onKeyUp={this.handleEnterPress} className='phone_field_wrap'>
                        <Input
                            id={field}
                            type='tel'
                            value={code}
                            name={field}
                            disabled={codeValid === 'valid'}
                            controlRef={(field) => (this.phoneCode = field)}
                            onFocus={this.checkInvalid}
                            onChange={this.handleCodeInput}
                            view='one-border'
                            state={codeValid === 'not_valid' ? 'error' : ''}
                            filled={!!code}
                            maxLength={isCodeWithFormatExp ? '7' : confirmationCodeLength}
                        />
                        <label
                            htmlFor={field}
                            className={classnames('registration__label', {
                                'registration__label-sm': isPhoneCallConfirmation
                            })}
                        >
                            {phoneEnterCodeLabel}
                        </label>
                        {isErrorPopupShow && <PhoneErrorPopup field={'phoneCode'} error={error} />}
                        {isMobile && !humanConfirmationDone && (
                            <ConfirmCodeBtnMobile
                                isPhoneConfirmationOpened={isPhoneConfirmationOpened}
                                canSwitchConfirmationMethod={canSwitchConfirmationMethod}
                                isMobile={isMobile}
                                showCodeHint={showCodeHint}
                                isShowCountDown={isShowCountDown}
                                restartTrigger={isPhoneCallConfirmation}
                                countDownDone={(codeValid !== 'valid' && !showHint) || (codeSentAgain && !showHint)}
                                isMobileRecallButtonExperiment={isMobileRecallButtonExperiment}
                                isPhoneCallConfirmation={isPhoneCallConfirmation}
                                btnLinks={hint.links}
                                status={codeValid}
                                confirmationRetryCountdown={confirmationRetryCountdown}
                                checkCode={this.checkCode}
                            />
                        )}{' '}
                        {/* https://st.yandex-team.ru/PASSP-20802 */}
                        {humanConfirmationDone && isMobile && (
                            <span>{i18n('Frontend.phone-confirm_message_confirmed')}</span>
                        )}
                        {!isMobile && (
                            <SendCodeHintPopup
                                isMobile={isMobile}
                                canSwitchConfirmationMethod={canSwitchConfirmationMethod}
                                error={phoneCodeError}
                                code={code}
                                showHint={showHint && codeValid !== 'valid'}
                                hint={hint}
                                canResendCode={!(codeSentAgain && isShowCountDown)}
                                setCodeSentStatus={this.setCodeSentStatus}
                            />
                        )}
                    </div>

                    {isFetching && (
                        <div className='fetching-block'>
                            <Spin progress={isFetching} size='m' />
                        </div>
                    )}

                    {!isMobile && <ConfirmCodeBtn status={codeValid} checkCode={this.checkCode} />}
                </div>
            </div>
        );
    }
}

export default connect(PhoneCodeInput.mapStateToProps)(PhoneCodeInput);

PhoneCodeInput.propTypes = {
    codeValid: PropTypes.bool,
    error: PropTypes.object,
    isMobile: PropTypes.bool.isRequired,
    isPhoneCallConfirmation: PropTypes.bool,
    isDiscardCallConfirmation: PropTypes.bool,
    changeNumber: PropTypes.func.isRequired,
    code: PropTypes.string,
    isPhoneCallConfirmationAvailable: PropTypes.bool,
    isMobileRecallButtonExperiment: PropTypes.bool,
    isShowCountDown: PropTypes.bool,
    isPhoneConfirmationOpened: PropTypes.bool,
    canSwitchConfirmationMethod: PropTypes.bool,
    isFetching: PropTypes.bool,
    showHint: PropTypes.bool,
    codeSentAgain: PropTypes.bool,
    hint: PropTypes.object,
    showCodeHint: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    validation: PropTypes.object.isRequired,
    confirmationRetryCountdown: PropTypes.number,
    callingNumberTemplate: PropTypes.string,
    confirmationCodeLength: PropTypes.number,
    isCodeWithFormatExp: PropTypes.bool
};
