import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import throttle from 'lodash/throttle';
import classnames from 'classnames';

import {Input} from '@components/Input';
import {PhoneErrorPopup, PhoneHintPopup} from './phone_field-popups.jsx';
import SendCode from './phone_send-code.jsx';
import {
    updateValues,
    updateErrorsValid,
    updatePhoneValuesToDefaults,
    updatePhoneStatesToDefaults,
    updateHintStatus,
    changePhoneConfirmationType,
    validatePhoneForCall,
    setCallConfirmationTimer,
    changeCallConfirmationProcess
} from '@blocks/actions/form';
import {updatePhoneHintStatus, updatePhoneHintText} from '@blocks/registration/actions';
import checkIfInvalid from '@blocks/registration/methods/checkIfInvalid';
import validatePhoneNumber from '@blocks/registration/methods/validatePhoneNumber';
import validatePassword from '@blocks/registration/methods/validatePassword';

const ChangePhoneNumberBtn = (props) => {
    const {isMobile, phoneCodeStatusSetToSent, changePhoneNumber, humanConfirmationDone} = props;

    if (isMobile && phoneCodeStatusSetToSent && !humanConfirmationDone) {
        return (
            <button className='field__btn' onClick={changePhoneNumber} type='button'>
                {i18n('Frontend.phone_short_change_number')}
            </button>
        );
    }
    return null;
};

ChangePhoneNumberBtn.propTypes = {
    isMobile: PropTypes.bool.isRequired,
    phoneCodeStatusSetToSent: PropTypes.bool.isRequired,
    changePhoneNumber: PropTypes.func.isRequired,
    humanConfirmationDone: PropTypes.bool.isRequired
};

export class PhoneConfirmation extends Component {
    constructor(props) {
        super(props);

        this.onFieldChange = this.onFieldChange.bind(this);
        this.handleFocus = this.handleFocus.bind(this);
        this.showHint = this.showHint.bind(this);
        this.changePhoneNumber = this.changePhoneNumber.bind(this);
        this.validatePhoneNumber = throttle(this.validatePhoneNumber.bind(this), 600);

        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        if (this.props.state === 'need_validation') {
            this.onFieldChange(this.props.value);
        }

        this.updateHintText();
        this.forceUpdate();
    }

    componentDidUpdate(prevProps) {
        const prevPhone = prevProps.value;
        const prevPhoneStatus = prevProps.phoneCodeStatus;

        if (prevPhone && prevPhoneStatus === 'code_sent' && !this.props.value) {
            this.phone.focus();
        }
    }

    validatePhoneNumber(value) {
        const {dispatch, isPhoneCallConfirmationAvailable} = this.props;

        if (isPhoneCallConfirmationAvailable) {
            dispatch(validatePhoneNumber(value));
        }
    }

    onFieldChange(event) {
        const {target = {}} = event;
        const {value} = target;
        const {isPasswordInvalid, passwordValue, errors, dispatch} = this.props;
        const fieldInfo = {
            field: 'phone',
            value
        };
        const isLikePhonePasswdError = errors.password && errors.password.code === 'likephonenumber';

        this.validatePhoneNumber(value);
        dispatch(updateErrorsValid(fieldInfo.field));
        dispatch(updateValues(fieldInfo));

        if (isPasswordInvalid && isLikePhonePasswdError) {
            dispatch(validatePassword(passwordValue));
        }
    }

    handleFocus() {
        const {dispatch, isPhoneCanBePrefilled, value} = this.props;

        if (isPhoneCanBePrefilled && !value) {
            dispatch(
                updateValues({
                    field: 'phone',
                    value: '+7 '
                })
            );
            setTimeout(() => {
                this.phone.setSelectionRange(3, 3);
            }, 1); // PASSP-21236
        }
        this.showHint();
    }

    showHint() {
        const {dispatch, isMobile, activeError, phoneConfirmation} = this.props;
        const {isPhoneHintShown} = phoneConfirmation;

        if (!isMobile) {
            if (activeError !== 'phone') {
                dispatch(updatePhoneHintStatus(!isPhoneHintShown));
            } else {
                dispatch(updatePhoneHintStatus(false));
                dispatch(checkIfInvalid('phone'));
            }
            // because of https://st.yandex-team.ru/PASSP-17389 (phone hint overlay password confirm error)
            dispatch(updateHintStatus(!isPhoneHintShown));
        }
    }

    updateHintText() {
        const {lang, tld, dispatch} = this.props;

        if (lang === 'ru' && (tld === 'by' || tld === 'uz')) {
            const updateTexts = {
                uz: i18n('Frontend.phone_hint_popup-text_uz'),
                by: i18n('Frontend.phone_hint_popup-text_by')
            };

            dispatch(updatePhoneHintText(updateTexts[tld]));
        }
    }

    changePhoneNumber() {
        const {isPhoneCallConfirmationAvailable, validation, dispatch} = this.props;

        dispatch(updatePhoneValuesToDefaults());
        dispatch(updatePhoneStatesToDefaults());
        dispatch(updateErrorsValid('phoneCode'));
        dispatch(changeCallConfirmationProcess(false));

        if (isPhoneCallConfirmationAvailable) {
            dispatch(changePhoneConfirmationType('sms'));
            dispatch(validatePhoneForCall(false));
        }

        if (validation.callConfirmationTimer) {
            clearTimeout(validation.callConfirmationTimer);
            dispatch(setCallConfirmationTimer(null));
        }
    }

    render() {
        const field = 'phone';
        const {
            value,
            state,
            phoneCodeStatus,
            phoneCode,
            activeError,
            errors,
            isPhoneCallConfirmationAvailable,
            isNoPhoneBtnAlwaysVisible,
            validation,
            isMobile,
            prefix,
            phoneConfirmation
        } = this.props;
        const phone = value;
        const notValid = state === 'not_valid';
        const phoneCodeStatusSetToSent = phoneCodeStatus === 'code_sent';
        const isSmsLimitExceededError = errors.phone.code === 'sms_limit.exceeded';
        const {isPhoneHintShown, hint} = phoneConfirmation;
        const showHint = isPhoneHintShown && !phoneCodeStatusSetToSent && activeError !== 'phone';
        const isShowSendCodeButton = Boolean(phone && phone.trim() !== '+7' && !isSmsLimitExceededError);
        const isShowCallButton = isShowSendCodeButton && !phoneCodeStatusSetToSent && isPhoneCallConfirmationAvailable;
        const isShowCallConfirmationBlock =
            validation.phoneConfirmationType === 'call' && phoneCodeStatusSetToSent && isPhoneCallConfirmationAvailable;

        return (
            <div
                className={classnames('phone__confirm-wrapper', {
                    'show-button': isShowSendCodeButton,
                    'no-phone-visible': isNoPhoneBtnAlwaysVisible,
                    'show-call-button': isShowCallButton,
                    'show-call-confirmation-block': isShowCallConfirmationBlock,
                    'confirm__fields-active': isMobile
                })}
            >
                <div
                    className={classnames('form__field', {
                        form__field_filled: phone || this.isLoading,
                        field__error: notValid,
                        field__valid: phoneCode === 'valid',
                        field__disabled: phoneCodeStatusSetToSent
                    })}
                >
                    <ChangePhoneNumberBtn
                        humanConfirmationDone={validation.humanConfirmationDone}
                        isMobile={isMobile}
                        phoneCodeStatusSetToSent={phoneCodeStatusSetToSent}
                        changePhoneNumber={this.changePhoneNumber}
                    />
                    <Input
                        id={field}
                        type='tel'
                        value={phone}
                        name={field}
                        disabled={phoneCodeStatusSetToSent}
                        controlRef={(phone) => (this.phone = phone)}
                        onFocus={this.handleFocus}
                        onBlur={this.showHint}
                        onChange={this.onFieldChange}
                        view='one-border'
                        state={notValid ? 'error' : ''}
                        filled={!!phone}
                    />
                    <label htmlFor={field} className='registration__label'>
                        {i18n('_AUTH_.field_label_reg-mobile')}
                    </label>
                    {notValid && errors.phone.code && (
                        <PhoneErrorPopup
                            isMobile={isMobile}
                            field='phone'
                            anchor={this.phone}
                            notValid={notValid && !phoneCodeStatusSetToSent}
                            error={errors}
                        />
                    )}
                    <PhoneHintPopup showHint={showHint} hint={hint} field={this.phone} />
                </div>

                <SendCode
                    isShowSendCodeButton={isShowSendCodeButton}
                    prefix={prefix}
                    changePhoneNumber={this.changePhoneNumber}
                    toggleConfirmationMethod={this.props.toggleConfirmationMethod}
                    ref={(sendComponent) => (this.sendComponent = sendComponent)}
                />
            </div>
        );
    }
}

PhoneConfirmation.propTypes = {
    dispatch: PropTypes.func.isRequired,
    toggleConfirmationMethod: PropTypes.func,
    validation: PropTypes.shape({
        phoneConfirmationType: PropTypes.string,
        isValidPhoneForCall: PropTypes.bool,
        magicConfirmationCode: PropTypes.string,
        callConfirmationTimer: PropTypes.number,
        isCallConfirmationInProcess: PropTypes.bool,
        isForceCheckedConfirmationCode: PropTypes.bool,
        humanConfirmationDone: PropTypes.bool,
        confirmationCodeLength: PropTypes.number
    }),
    prefix: PropTypes.string,
    isMobile: PropTypes.bool,
    isPhoneCanBePrefilled: PropTypes.bool,
    isPhoneCallConfirmationAvailable: PropTypes.bool.isRequired,
    value: PropTypes.string.isRequired,
    state: PropTypes.string,
    phoneCodeStatus: PropTypes.string,
    phoneCode: PropTypes.string,
    activeError: PropTypes.string,
    errors: PropTypes.object.isRequired,
    lang: PropTypes.string,
    tld: PropTypes.string,
    isPasswordInvalid: PropTypes.bool,
    isNoPhoneBtnAlwaysVisible: PropTypes.bool,
    passwordValue: PropTypes.string,
    phoneConfirmation: PropTypes.object
};

function mapStateToProps(state) {
    const {
        phoneConfirmation,
        settings: {uatraits: ua, language: lang, tld},
        form: {
            states,
            values,
            errors,
            validation,
            humanConfirmation,
            canSwitchConfirmationMethod,
            isPhoneCallConfirmationAvailable,
            isForcePhoneCallConfirmation,
            isDiscardCallConfirmation,
            isPhoneCanBePrefilled
        },
        experimentsList
    } = state;
    const activeError = errors.active;
    const isMobile = ua.isMobile && !ua.isTablet;
    const isNoPhoneBtnAlwaysVisible = experimentsList.isNoPhoneBtnShowed;

    return {
        phoneConfirmation,
        activeError,
        tld,
        lang,
        value: values.phone,
        state: states.phone,
        passwordValue: values.password,
        isPasswordInvalid: states.password && states.password === 'not_valid',
        phoneCodeStatus: states.phoneCodeStatus,
        phoneCode: states.phoneCode,
        phoneCodeValue: values.phoneCode,
        validation,
        isMobile,
        canSwitchConfirmationMethod,
        isPhoneCallConfirmationAvailable,
        isForcePhoneCallConfirmation,
        isDiscardCallConfirmation,
        isPhoneCanBePrefilled,
        isNoPhoneBtnAlwaysVisible,
        humanConfirmation,
        errors
    };
}

export default connect(mapStateToProps)(PhoneConfirmation);
