import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import ReactDOM from 'react-dom';
import {connect} from 'react-redux';
import submitPhoneConfirmationCode from '@blocks/registration/methods/submitPhoneConfirmationCode';
import validatePassword from '@blocks/registration/methods/validatePassword';
import validatePhoneNumber from '@blocks/registration/methods/validatePhoneNumber';
// eslint-disable-next-line max-len
import sendMetricsForShowCallConfirmationBtns from '@blocks/registration/methods/sendMetricsForShowCallConfirmationBtns';
import sendMetricsForSendingConfirmationCode from '@blocks/registration/methods/sendMetricsForSendingConfirmationCode';
import sendPhoneCode from '@blocks/registration/methods/phoneConfirmation/sendPhoneCode';
import {updateErrors, setCallConfirmationTimer, changeCallConfirmationProcess} from '@blocks/actions/form';
import {
    REGISTRATION_GOAL_PREFIX,
    togglePhoneConfirmationPopup,
    updatePhoneCodeHintText,
    updatePhoneCodeSendAgain,
    updatePhoneCodeHintStatus
} from '@blocks/registration/actions';

import metrics from '@blocks/metrics';
import {SendCodeBtns} from './send_code_btns';
import PhoneCodeInput from './PhoneCodeInput.jsx';
import {hasExp} from '@blocks/selectors';

export class SendCode extends PureComponent {
    static mapStateToProps(state) {
        const {
            settings: {uatraits},
            customs: {hideRegistration: {noPhoneLink: customsNoPhoneLink = false} = {}} = {},
            common: {origin = null} = {},
            form,
            isPhoneConfirmationOpened,
            isPhoneConfirmationExperiment,
            phoneConfirmation
        } = state;
        const {
            validation,
            states,
            errors,
            values,
            humanConfirmation,
            isPhoneCallConfirmationAvailable,
            isForcePhoneCallConfirmation,
            isDiscardCallConfirmation,
            isMobileRecallBtnRemoved,
            isMobileCallConfirmationOnly,
            canSwitchConfirmationMethod,
            confirmationRetryCountdown
        } = form;
        const isMobile = uatraits.isMobile && !uatraits.isTablet;
        const isCodeSent = states.phoneCodeStatus === 'code_sent';
        const isConfirmBtnTextExperiment = isCodeSent && isPhoneConfirmationExperiment;
        const isOldregistrationNoPhone = hasExp(state, 'oldregistration-nophone');

        return {
            isMobile,
            validation,
            error: errors,
            phone: values.phone,
            password: values.password,
            isFetching: humanConfirmation.isFetching,
            code: values.phoneCode,
            codeStatus: states.phoneCodeStatus,
            isCodeSent,
            hidePhoneLink: customsNoPhoneLink || (!origin && hasExp(state, 'exp-registration-no-phone-link-no-origin')),
            codeValid: states.phoneCode,
            isPhoneCallConfirmationAvailable,
            isForcePhoneCallConfirmation,
            isDiscardCallConfirmation,
            canSwitchConfirmationMethod: canSwitchConfirmationMethod && !isOldregistrationNoPhone,
            isPhoneConfirmationOpened,
            isConfirmBtnTextExperiment,
            phoneConfirmation,
            confirmationRetryCountdown,
            isMobileRecallButtonExperiment: isMobileRecallBtnRemoved && isMobile,
            isMobileNoSmsExperiment: isMobileCallConfirmationOnly && isMobile
        };
    }

    constructor(props) {
        super(props);

        this.showCodeHint = this.showCodeHint.bind(this);
        this.sendCode = this.sendCode.bind(this);
        this.setCodeSentStatus = this.setCodeSentStatus.bind(this);
        this.setupHintText = this.setupHintText.bind(this);
        this.setInputFocus = this.setInputFocus.bind(this);
        this.changeCallConfirmationProcess = this.changeCallConfirmationProcess.bind(this);
        this.renderPhoneConfirmation = this.renderPhoneConfirmation.bind(this);
        this.toggleConfirmationMethod = this.toggleConfirmationMethod.bind(this);
        this.closeConfirmationPopup = this.closeConfirmationPopup.bind(this);
        this.handleHintLink = this.handleHintLink.bind(this);

        this.focusTimer = null;
    }

    componentDidMount() {
        this.setupHintText();
    }

    componentDidUpdate(prevProps) {
        const {
            validation: {phoneConfirmationType, isValidPhoneForCall},
            error,
            dispatch
        } = this.props;

        if (this.props.codeStatus !== prevProps.codeStatus) {
            this.setInputFocus();
            this.props.dispatch(updatePhoneCodeHintStatus(false));
        }

        if (phoneConfirmationType !== prevProps.validation.phoneConfirmationType) {
            this.setupHintText();
            this.props.dispatch(updatePhoneCodeHintStatus(false));
            this.props.dispatch(updateErrors({field: 'phone', error: '', active: false}));
            this.props.dispatch(updateErrors({field: 'phoneCode', error: '', active: false}));

            return;
        }

        if (isValidPhoneForCall !== prevProps.validation.isValidPhoneForCall) {
            this.setupHintText();

            if (isValidPhoneForCall) {
                dispatch(sendMetricsForShowCallConfirmationBtns());
            }
        }

        const isShowPhoneCodeError =
            error.phoneCode.code &&
            prevProps.error.phoneCode.code !== error.phoneCode.code &&
            ['code.empty', 'code.invalid', 'confirmations_limit.exceeded', 'code.missingvalue', 'missingvalue'].indexOf(
                error.phoneCode.code
            ) === -1;

        if (isShowPhoneCodeError) {
            this.showCodeHint(true);
        }
    }

    componentWillUnmount() {
        if (this.focusTimer) {
            clearTimeout(this.focusTimer);
        }
    }

    changeCallConfirmationProcess(process) {
        this.props.dispatch(changeCallConfirmationProcess(process));
    }

    handleHintLink(method, ...methodArgs) {
        this.props.dispatch(method(...methodArgs));
    }

    setInputFocus() {
        if (this.focusTimer) {
            clearTimeout(this.focusTimer);
        }

        this.focusTimer = setTimeout(() => {
            if (!this.phoneCode) {
                return;
            }

            // eslint-disable-next-line react/no-find-dom-node
            const field = ReactDOM.findDOMNode(this.phoneCode).querySelector('input');

            field.focus();
            field.click();
        }, 300);
    }

    setupHintText() {
        const {
            validation: {phoneConfirmationType, isValidPhoneForCall},
            isPhoneCallConfirmationAvailable,
            canSwitchConfirmationMethod,
            isMobileRecallButtonExperiment,
            isMobileNoSmsExperiment,
            hidePhoneLink,
            dispatch
        } = this.props;

        let hintTitle = i18n('Frontend.phone_code_not_received');

        let hintText = i18n('Frontend.phone_code_info');

        let hintLinks = [
            {
                text: i18n('_AUTH_.phone_send_code_again'),
                method: this.handleHintLink.bind(null, sendPhoneCode, true, false),
                name: 'sendSmsAgain'
            }
        ];

        if (!hidePhoneLink) {
            hintLinks.push({
                text: i18n('Frontend.no_phone_reglink'),
                method: this.toggleConfirmationMethod.bind(this),
                name: 'toggleConfirmationMethod'
            });
        }

        if (!canSwitchConfirmationMethod) {
            hintLinks.splice(1);
        }

        if (phoneConfirmationType === 'call') {
            hintTitle = i18n('Frontend.call_did_not_arrive');
            hintText = '';
            hintLinks = [];

            if (!isMobileNoSmsExperiment) {
                hintLinks.push({
                    text: i18n('Frontend.send_confirmation_code_in_sms'),
                    method: this.handleHintLink.bind(null, sendPhoneCode, false, false),
                    name: 'confirmWithSms'
                });
            }

            if (canSwitchConfirmationMethod && !hidePhoneLink) {
                hintLinks.unshift({
                    text: i18n('Frontend.without_phone_registration'),
                    method: this.toggleConfirmationMethod.bind(this),
                    name: 'toggleConfirmationMethod'
                });
            }

            if (isPhoneCallConfirmationAvailable && isValidPhoneForCall && !isMobileRecallButtonExperiment) {
                hintLinks.unshift({
                    text: i18n('_AUTH_.confirmation_call_repeat'),
                    method: this.handleHintLink.bind(null, sendPhoneCode, true, true),
                    name: 'makeCallAgain'
                });
            }
        }

        dispatch(updatePhoneCodeSendAgain(false));
        dispatch(
            updatePhoneCodeHintText({
                title: hintTitle,
                text: hintText,
                links: hintLinks
            })
        );
    }

    showCodeHint(showHint) {
        const {dispatch, isPhoneCallConfirmationAvailable, validation, phone} = this.props;
        const isNeedPhoneValidation = isPhoneCallConfirmationAvailable && validation.phoneConfirmationType === 'call';

        if (isNeedPhoneValidation) {
            dispatch(validatePhoneNumber(phone));
        }

        this.props.dispatch(updatePhoneCodeHintStatus(showHint));
    }

    toggleConfirmationMethod() {
        const prefix = this.props.prefix || REGISTRATION_GOAL_PREFIX;

        this.props.toggleConfirmationMethod();
        this.props.dispatch(togglePhoneConfirmationPopup(false));

        metrics.send(['Подтверждение телефона', 'Регистрируйтесь без телефона']);
        metrics.goal(`${prefix}_phoneconfirm_switch_method`);
    }

    sendCode(sendAgain = false, isCallConfirmation = false) {
        const {
            isMobile,
            password,
            dispatch,
            phone,
            isCodeSent,
            validation: {phoneConfirmationType, callConfirmationTimer}
        } = this.props;
        const prefix = this.props.prefix || REGISTRATION_GOAL_PREFIX;
        const isSendCodeShouldBeSkipped = isMobile && !sendAgain && isCodeSent && phoneConfirmationType !== 'call';

        this.showCodeHint(false);

        if (isMobile) {
            dispatch(togglePhoneConfirmationPopup(true));
        }

        if (phoneConfirmationType === 'call') {
            if (callConfirmationTimer) {
                clearTimeout(callConfirmationTimer);
            }

            dispatch(setCallConfirmationTimer(null));
        }

        if (sendAgain && isCallConfirmation) {
            dispatch(updateErrors({field: 'phone', error: '', active: false}));
            dispatch(updateErrors({field: 'phoneCode', error: '', active: false}));
        }

        if (!isSendCodeShouldBeSkipped) {
            if (typeof password !== 'undefined' && prefix.indexOf('complete_lite') !== 0) {
                dispatch(validatePassword(password, 'phone', isCallConfirmation));
            } else {
                dispatch(submitPhoneConfirmationCode(phone, isCallConfirmation));
            }

            if (sendAgain && isCodeSent) {
                this.setCodeSentStatus();
            }

            dispatch(sendMetricsForSendingConfirmationCode(sendAgain, isCallConfirmation));
        }
    }

    setCodeSentStatus() {
        const {dispatch, isPhoneCallConfirmationAvailable, validation, phone, phoneConfirmation} = this.props;
        const {isCodeSentAgain} = phoneConfirmation;
        const isNeedPhoneValidation = isPhoneCallConfirmationAvailable && validation.phoneConfirmationType === 'call';

        if (isNeedPhoneValidation) {
            dispatch(validatePhoneNumber(phone));
        }

        dispatch(updatePhoneCodeSendAgain(!isCodeSentAgain));
    }

    closeConfirmationPopup() {
        this.props.dispatch(togglePhoneConfirmationPopup(false));
    }

    renderPhoneConfirmation() {
        const {
            isMobile,
            phone,
            codeValid,
            error,
            isFetching,
            canSwitchConfirmationMethod,
            toggleConfirmationMethod,
            isPhoneCallConfirmationAvailable,
            isForcePhoneCallConfirmation,
            changePhoneNumber,
            isShowSendCodeButton,
            isPhoneConfirmationOpened,
            isConfirmBtnTextExperiment,
            isCodeSent,
            phoneConfirmation,
            validation: {phoneConfirmationType, isValidPhoneForCall},
            confirmationRetryCountdown,
            dispatch
        } = this.props;
        const isPhoneCallConfirmation = phoneConfirmationType === 'call';
        const isCodeInvalid = codeValid !== 'valid';
        const {isCodeSentAgain, isCodeHintShown, codeHint} = phoneConfirmation;
        const isShowCountDown =
            (isCodeSent && isCodeInvalid && !isCodeHintShown) || (isCodeSentAgain && !isCodeHintShown);
        const isErrorPopupShow = !isCodeHintShown || isMobile;
        const isSendCodeButtonsShouldBeRendered = !isCodeSent || (isMobile && !isPhoneConfirmationOpened && isCodeSent);

        if (isSendCodeButtonsShouldBeRendered) {
            return (
                <SendCodeBtns
                    isMobile={isMobile}
                    dispatch={dispatch}
                    isConfirmBtnTextExperiment={isConfirmBtnTextExperiment}
                    isPhoneConfirmationOpened={isPhoneConfirmationOpened}
                    isShowCountDown={isShowCountDown}
                    phone={phone}
                    error={error}
                    isShowSendCodeButton={isShowSendCodeButton}
                    isPhoneCallConfirmationAvailable={isPhoneCallConfirmationAvailable}
                    isForcePhoneCallConfirmation={isForcePhoneCallConfirmation}
                    canSwitchConfirmationMethod={canSwitchConfirmationMethod}
                    isValidPhoneForCall={isValidPhoneForCall}
                    isFetching={isFetching}
                    toggleConfirmationMethod={toggleConfirmationMethod}
                />
            );
        }

        if (!isCodeInvalid && isPhoneCallConfirmation) {
            return (
                <div className='phone__code-btn'>
                    <span>{i18n('Frontend.phone-confirm_message_confirmed')}</span>
                </div>
            );
        }

        return (
            <PhoneCodeInput
                isPhoneCallConfirmation={isPhoneCallConfirmation}
                showCodeHint={this.showCodeHint}
                isErrorPopupShow={isErrorPopupShow}
                isShowCountDown={isShowCountDown}
                showHint={isCodeHintShown}
                codeSentAgain={isCodeSentAgain}
                changeNumber={changePhoneNumber}
                confirmationRetryCountdown={confirmationRetryCountdown}
                hint={codeHint}
            />
        );
    }

    render() {
        const {validation, isPhoneConfirmationOpened, isCodeSent} = this.props;
        const {humanConfirmationDone} = validation;
        const isConfirmationShowedInSeparateBlock = isPhoneConfirmationOpened && !humanConfirmationDone && isCodeSent;

        return (
            <div className={isConfirmationShowedInSeparateBlock ? 'phone-confirmations__block' : ''}>
                {isConfirmationShowedInSeparateBlock && (
                    <button type='button' className='phone-confirmation__close' onClick={this.closeConfirmationPopup}>
                        {i18n('Frontend.close')}
                    </button>
                )}
                {this.renderPhoneConfirmation()}
            </div>
        );
    }
}

SendCode.propTypes = {
    dispatch: PropTypes.func.isRequired,
    toggleConfirmationMethod: PropTypes.func,
    code: PropTypes.string.isRequired,
    phone: PropTypes.string.isRequired,
    password: PropTypes.string,
    codeStatus: PropTypes.string.isRequired,
    codeValid: PropTypes.string,
    error: PropTypes.object.isRequired,
    isShowSendCodeButton: PropTypes.bool,
    validation: PropTypes.shape({
        phoneConfirmationType: PropTypes.string,
        isValidPhoneForCall: PropTypes.bool,
        magicConfirmationCode: PropTypes.string,
        callConfirmationTimer: PropTypes.number,
        isCallConfirmationInProcess: PropTypes.bool,
        isForceCheckedConfirmationCode: PropTypes.bool,
        humanConfirmationDone: PropTypes.bool,
        confirmationCodeLength: PropTypes.number
    }),
    changePhoneNumber: PropTypes.func.isRequired,
    isFetching: PropTypes.bool,
    canSwitchConfirmationMethod: PropTypes.bool.isRequired,
    isPhoneCallConfirmationAvailable: PropTypes.bool.isRequired,
    isForcePhoneCallConfirmation: PropTypes.bool.isRequired,
    isDiscardCallConfirmation: PropTypes.bool.isRequired,
    prefix: PropTypes.string,
    isMobile: PropTypes.bool,
    isPhoneConfirmationOpened: PropTypes.bool,
    isConfirmBtnTextExperiment: PropTypes.bool,
    isMobileNoSmsExperiment: PropTypes.bool,
    isMobileRecallButtonExperiment: PropTypes.bool,
    isCodeSent: PropTypes.bool,
    hidePhoneLink: PropTypes.bool,
    phoneConfirmation: PropTypes.object,
    confirmationRetryCountdown: PropTypes.number
};

SendCode.contextTypes = {
    prefix: PropTypes.string
};

export default connect(SendCode.mapStateToProps)(SendCode);
