import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {Input} from '@components/Input';
import LoginSuggest from './suggest.jsx';
import LoginField from '@blocks/registration/mobile/components/login_field.jsx';
import classnames from 'classnames';
import validateLogin from '@blocks/registration/methods/validateLogin';
import getSuggestedLogins from '@blocks/registration/methods/getSuggestedLogins';
import {changeActiveField, setFieldErrorActive} from '@blocks/actions/form';

import debounce from 'lodash/debounce';

export class RegistrationLoginInput extends PureComponent {
    static mapStateToProps(state) {
        const {
            form,
            settings: {uatraits}
        } = state;
        const {values, states, errors, activeField} = form;
        const isMobile = uatraits.isMobile && !uatraits.isTablet;

        return {
            value: values.login,
            state: states.login,
            error: errors.active,
            nameIsFilled: values.firstname && values.lastname,
            activeField,
            isMobile
        };
    }
    constructor(props) {
        super(props);

        this.state = {
            isActive: false,
            isValid: false,
            isFocused: false,
            isNavigateByTab: false
        };
        this.handleInput = this.handleInput.bind(this);
        this.handleFocus = this.handleFocus.bind(this);
        this.handleFocusout = this.handleFocusout.bind(this);
        this.handleValidation = debounce(this.handleValidation.bind(this), 600);
        this.handleTabNavigation = this.handleTabNavigation.bind(this);

        this.timeOut = 2000;
        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        if (this.props.state === 'need_validation') {
            this.handleValidation(this.props.value);
        }

        this.forceUpdate();
    }

    handleInput(event) {
        const {target = {}} = event;
        const {value} = target;
        const {updateUserField} = this.props;

        updateUserField(event);
        this.handleValidation(value);
    }

    handleValidation(value) {
        this.props.dispatch(validateLogin(value));
    }

    handleFocus(e) {
        const {value, state, dispatch, nameIsFilled} = this.props;
        const fieldIsValid = state === 'valid';

        e.target.removeAttribute('readonly');

        dispatch(changeActiveField('login'));
        dispatch(setFieldErrorActive(''));

        this.setState({
            isActive: true,
            isFocused: true,
            isValid: fieldIsValid
        });

        if (value) {
            dispatch(validateLogin(value));
        }

        if (nameIsFilled) {
            dispatch(getSuggestedLogins());
        }
    }

    handleFocusout() {
        const {dispatch} = this.props;
        const fieldIsValid = this.props.state === 'valid';

        this.setState(() => ({isFocused: false}));

        setTimeout(
            function() {
                if (fieldIsValid || (!this.state.isFocused && !this.state.isNavigateByTab)) {
                    this.setState({isActive: false, isValid: fieldIsValid});

                    if (this.props.activeField === 'login') {
                        dispatch(changeActiveField(''));
                    }
                }
            }.bind(this),
            this.timeOut
        );
    }

    handleTabNavigation(event) {
        if (event && event.keyCode === 9) {
            this.setState({isNavigateByTab: true});
        } else {
            this.setState({isNavigateByTab: false, isActive: false, isValid: this.props.state === 'valid'});
        }
    }

    render() {
        const {activeField, state, value, isMobile, isAutoComplete} = this.props;
        const text = value;
        const isHintActive = this.props.hintActive;
        const notValid = state === 'not_valid';
        const blockCssClasses = classnames('form__field form__field_wide', {
            form__field_filled: text || this.isLoading,
            field__error: notValid,
            field__valid: state === 'valid'
        });

        if (isMobile) {
            return <LoginField isAutoComplete={isAutoComplete} />;
        }

        return (
            <div className={blockCssClasses}>
                {isAutoComplete === false && (
                    <div className='registration__chrome_bag_autocomplete'>
                        <input type='input' name='login' tabIndex='-1' />
                    </div>
                )}
                <Input
                    id='login'
                    name='login'
                    value={text}
                    autoComplete={isAutoComplete === false ? 'off' : 'username'}
                    controlRef={(login) => (this.login = login)}
                    onChange={this.handleInput}
                    onFocus={this.handleFocus}
                    onBlur={this.handleFocusout}
                    view='one-border'
                    state={notValid ? 'error' : ''}
                    filled={!!value || this.isLoading}
                />
                <label htmlFor='login' className='registration__label'>
                    {i18n('_AUTH_.field_login')}
                </label>
                <LoginSuggest
                    login={text}
                    anchor={this.login}
                    handleTabNavigation={this.handleTabNavigation}
                    isHintActive={isHintActive}
                    fieldValid={this.state.isValid}
                    fieldActive={this.state.isActive && activeField === 'login'}
                />
            </div>
        );
    }
}

RegistrationLoginInput.propTypes = {
    value: PropTypes.string.isRequired,
    state: PropTypes.string.isRequired,
    error: PropTypes.string.isRequired,
    nameIsFilled: PropTypes.string,
    hintActive: PropTypes.bool,
    activeField: PropTypes.string.isRequired,
    updateUserField: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    isMobile: PropTypes.bool,
    isAutoComplete: PropTypes.bool
};

export default connect(RegistrationLoginInput.mapStateToProps)(RegistrationLoginInput);
