import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {Button} from '@components/Button';
import {Spin} from '@components/Spin';
import FieldPopup from '@blocks/registration/desktop/errorPopup/error_popup.jsx';
import classnames from 'classnames';
import {REGISTRATION_GOAL_PREFIX} from '@blocks/registration/actions';
import {updateValues, updateStates, updateErrorsValid} from '@blocks/actions/form';
import metrics from '@blocks/metrics';

const LoginsList = ({logins, onSelectLogin, isFetching}) => {
    if (logins.length !== 0) {
        return (
            <div className='suggest__logins'>
                <strong className='suggest__status-text' tabIndex='4'>
                    {i18n('Frontend.login_free_list')}
                </strong>
                <ul className='logins__list'>
                    {logins.map((login) => (
                        <li className='registration__pseudo-link' key={login}>
                            <label
                                htmlFor={login}
                                tabIndex='0'
                                title={login}
                                onKeyDown={onSelectLogin}
                                data-login={login}
                            >
                                <input type='radio' onChange={onSelectLogin} id={login} />
                                {login}
                            </label>
                        </li>
                    ))}
                </ul>
            </div>
        );
    }
    if (logins.length === 0 && isFetching) {
        return (
            <div className='suggest__spinner'>
                <Spin progress={true} size='m' />
            </div>
        );
    }
    return null;
};

LoginsList.propTypes = {
    logins: PropTypes.array,
    onSelectLogin: PropTypes.func.isRequired,
    isFetching: PropTypes.bool
};

const Suggest = (props) => {
    const showMoreButton = props.loginsList && props.loginsList.length > 7;

    if (props.fieldActive && props.loginsList.length) {
        return (
            <div>
                {props.errorDescription && <span className='suggest__description-text'>{props.errorDescription}</span>}
                <LoginsList
                    logins={props.loginsToShow}
                    isFetching={props.isFetching}
                    onSelectLogin={props.setSelectedLogin}
                />
                {showMoreButton && (
                    <ButtonShowMore
                        showAll={props.showAll}
                        loginsList={props.loginsList}
                        showMoreLogins={props.showMoreLogins}
                    />
                )}
            </div>
        );
    }
    return null;
};

Suggest.propTypes = {
    fieldActive: PropTypes.bool.isRequired,
    errorDescription: PropTypes.string,
    loginsToShow: PropTypes.array,
    isFetching: PropTypes.bool,
    setSelectedLogin: PropTypes.func,
    showAll: PropTypes.bool,
    loginsList: PropTypes.array,
    showMoreLogins: PropTypes.func
};

const ButtonShowMore = (props) => {
    if (!props.showAll && props.loginsList.length > 0) {
        return (
            <div className='login__suggest-button'>
                <Button
                    view='pseudo'
                    size='s'
                    width='max'
                    tabIndex='0'
                    onKeyUp={props.showMoreLogins}
                    onClick={props.showMoreLogins}
                >
                    {i18n('Frontend.suggest_more_logins')}
                </Button>
            </div>
        );
    }
    return null;
};

ButtonShowMore.propTypes = {
    showAll: PropTypes.bool,
    loginsList: PropTypes.array,
    showMoreLogins: PropTypes.func
};

class LoginSuggest extends PureComponent {
    static mapStateToProps(state) {
        const logins = state.logins;
        const form = state.form;
        const activeError = form.errors.active || '';
        const loginError = form.errors.login;
        const errorDescription = loginError.descriptionText;
        const loginState = form.states.login;

        return {
            logins,
            activeError,
            errorDescription,
            loginError,
            loginState
        };
    }

    constructor(props) {
        super(props);

        this.setSelectedLogin = this.setSelectedLogin.bind(this);
        this.showMoreLogins = this.showMoreLogins.bind(this);
        this.setLoginsToShow = this.setLoginsToShow.bind(this);
        this.setActive = this.setActive.bind(this);

        this.state = {
            showList: false,
            showSuggestedLogins: false,
            loginFromSuggestPicked: false,
            showAll: false
        };
    }

    setActive() {
        this.setState({
            showList: true
        });
    }

    setSelectedLogin(event) {
        const target = event.target;
        const isKeyEvent = event.keyCode && event.keyCode === 13;
        const selectedLogin = isKeyEvent ? target.dataset.login : target.id;
        const dispatch = this.props.dispatch;
        const prefix = this.context.prefix || REGISTRATION_GOAL_PREFIX;

        if (isKeyEvent) {
            event.preventDefault();
            this.props.handleTabNavigation();
        }

        if (selectedLogin) {
            dispatch(updateValues({field: 'login', value: selectedLogin}));
            dispatch(updateStates({field: 'login', status: 'valid'}));
            dispatch(updateErrorsValid('login'));

            metrics.send(['Саджест логина', 'Выбор логина из списка']);
            metrics.goal(`${prefix}_suggest_selected`);

            this.setState({showSuggestedLogins: false, loginFromSuggestPicked: true, showAll: false, showList: false});

            if (isKeyEvent) {
                document.querySelector('#password').focus();
            }
        }
    }

    showMoreLogins(event) {
        const prefix = this.context.prefix || REGISTRATION_GOAL_PREFIX;

        this.setState({
            showAll: true,
            showList: true
        });

        if (event && event.keyCode === 9) {
            this.props.anchor.focus();
        }
        metrics.send(['Саджест логина', 'Ещё 5 логинов']);
        metrics.goal(`${prefix}_suggest_more`);
    }

    setLoginsToShow() {
        const {loginsList} = this.props.logins;

        if (loginsList && loginsList.length > 7) {
            return loginsList.slice(0, 5);
        }

        return loginsList;
    }

    render() {
        const {
            errorDescription,
            loginError,
            loginState,
            logins,
            fieldActive,
            isHintActive,
            fieldValid,
            handleTabNavigation
        } = this.props;
        const {loginsList = [], isFetching} = logins;
        const isVisible = loginsList.length > 0 && fieldActive && !isHintActive;
        const loginsToShow = this.state.showAll ? loginsList : this.setLoginsToShow();
        const loginIsValid = loginState === 'valid';
        const blockClass = loginsList.length > 0 ? 'form__login-suggest' : 'form__login-message';

        if (loginIsValid && fieldActive && !fieldValid) {
            return (
                <FieldPopup visible={true}>
                    <div className='form__login-message form__popup-text'>
                        <strong className='suggest__status-text suggest__valid'>
                            {i18n('common.login-available')}
                        </strong>
                    </div>
                </FieldPopup>
            );
        }

        return (
            <FieldPopup visible={isVisible && !loginIsValid} fieldName='login'>
                <div className='form__popup-error form__popup-text' data-t='login-error' role='alert'>
                    <div className={blockClass} onKeyUp={handleTabNavigation}>
                        {loginError.text && (
                            <strong className={classnames('suggest__status-text', {'error-message': loginError.text})}>
                                {loginError.text}
                            </strong>
                        )}
                        <Suggest
                            fieldActive={fieldActive || this.state.showList}
                            showMoreLogins={this.showMoreLogins}
                            loginsToShow={loginsToShow}
                            showAll={this.state.showAll}
                            loginsList={loginsList}
                            isFetching={isFetching}
                            errorDescription={errorDescription}
                            setSelectedLogin={this.setSelectedLogin}
                        />
                    </div>
                </div>
            </FieldPopup>
        );
    }
}

LoginSuggest.propTypes = {
    activeError: PropTypes.string,
    anchor: PropTypes.object,
    dispatch: PropTypes.func.isRequired,
    fieldActive: PropTypes.bool,
    fieldValid: PropTypes.bool,
    fieldFocused: PropTypes.bool,
    isHintActive: PropTypes.bool,
    login: PropTypes.string,
    loginError: PropTypes.shape({
        code: PropTypes.string,
        text: PropTypes.string
    }),
    errorDescription: PropTypes.string,
    loginState: PropTypes.string,
    logins: PropTypes.shape({
        loginsList: PropTypes.array,
        isFetching: PropTypes.bool
    }),
    handleTabNavigation: PropTypes.func.isRequired
};

LoginSuggest.contextTypes = {
    prefix: PropTypes.string
};

export default connect(LoginSuggest.mapStateToProps)(LoginSuggest);
