import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Input} from '@components/Input';
import FieldPopup from '@blocks/registration/desktop/errorPopup/error_popup.jsx';
import classnames from 'classnames';
import {focusInput} from '@blocks/utils';
import checkIfInvalid from '@blocks/registration/methods/checkIfInvalid';
import validateName from '@blocks/registration/methods/validateName';
import {changeActiveField, setFieldErrorActive} from '@blocks/actions/form';

export default class RegistrationNameInput extends PureComponent {
    constructor(props) {
        super(props);

        this.validate = this.validate.bind(this);
        this.handleFocus = this.handleFocus.bind(this);
        this.checkInvalid = this.checkInvalid.bind(this);
        this.checkFieldPopupVisible = this.checkFieldPopupVisible.bind(this);

        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        if (!this.props.isAutoFocuseDisabled) {
            focusInput(this.field);
        }

        if (this.props.state === 'need_validation') {
            this.validate();
        }

        this.forceUpdate();
    }

    validate() {
        const inputVal = this.props.value;

        this.props.dispatch(validateName(inputVal, 'firstname'));
    }

    checkInvalid() {
        this.props.dispatch(checkIfInvalid('firstname'));
    }

    handleFocus() {
        const {dispatch} = this.props;

        dispatch(setFieldErrorActive(''));
        dispatch(changeActiveField('firstname'));
        dispatch(checkIfInvalid('firstname'));
    }

    checkFieldPopupVisible() {
        const {isMobile, state, activeError} = this.props;
        const isNotValid = state === 'not_valid';

        if (isMobile) {
            return isNotValid;
        }
        return isNotValid && activeError === 'firstname';
    }

    render() {
        const {updateUserField, value, state, error} = this.props;
        const field = 'firstname';
        const isNotValid = state === 'not_valid';
        const isPopupVisible = this.checkFieldPopupVisible();

        return (
            <div
                className={classnames('form__field', {
                    form__field_filled: value || this.isLoading,
                    field__error: isNotValid,
                    field__valid: state === 'valid'
                })}
            >
                <Input
                    controlRef={(ref) => (this.field = ref)}
                    id={field}
                    value={value}
                    name={field}
                    onChange={updateUserField}
                    onFocus={this.handleFocus}
                    onBlur={this.validate}
                    view='one-border'
                    state={isNotValid ? 'error' : ''}
                    filled={!!value || this.isLoading}
                />
                <label htmlFor='firstname' className='registration__label'>
                    {i18n('_AUTH_.field_firstname')}
                </label>
                <FieldPopup fieldName='firstname' visible={isPopupVisible}>
                    <div className='form__popup-error' data-t='name-error' role='alert'>
                        <div className='error-message'>{error.text}</div>
                    </div>
                </FieldPopup>
            </div>
        );
    }
}

RegistrationNameInput.propTypes = {
    value: PropTypes.string.isRequired,
    state: PropTypes.string.isRequired,
    activeError: PropTypes.string,
    activeField: PropTypes.string.isRequired,
    error: PropTypes.object.isRequired,
    updateUserField: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    isAutoFocuseDisabled: PropTypes.bool,
    isMobile: PropTypes.bool
};
