import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Input} from '@components/Input';
import FieldPopup from '../../desktop/errorPopup/error_popup.jsx';
import classnames from 'classnames';
import checkIfInvalid from '../../methods/checkIfInvalid';
import validatePasswordConfirm from '../../methods/validatePasswordConfirm';

export default class RegistrationPasswordConfirm extends PureComponent {
    constructor(props) {
        super(props);

        this.state = {
            forceHidden: false
        };

        this.handleInput = this.handleInput.bind(this);
        this.validate = this.validate.bind(this);
        this.checkInvalid = this.checkInvalid.bind(this);

        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        if (this.props.passwordConfirmState === 'need_validation') {
            this.validate();
        }

        this.forceUpdate();
    }

    handleInput(event) {
        const {target = {}} = event;
        const {value} = target;
        const {updateField} = this.props;

        updateField(event);
        this.validate(value);
        if (!this.state.forceHidden) {
            this.setState({
                forceHidden: true
            });
        }
    }

    validate(cur) {
        const {passwordValue, dispatch} = this.props;
        const isBlur = typeof cur !== 'string';
        const value = isBlur ? this.props.value : cur;

        dispatch(validatePasswordConfirm(value, passwordValue));

        if (isBlur) {
            this.setState({
                forceHidden: false
            });
        }
    }

    checkInvalid() {
        this.props.defineActiveField('password_confirm');
        this.props.dispatch(checkIfInvalid('password_confirm'));
    }

    render() {
        const field = 'password_confirm';
        const {value, passwordConfirmState, passwordConfirmError, activeField, fieldType, hintActive} = this.props;
        const text = value;
        const notValid = passwordConfirmState === 'not_valid';
        const isActive = activeField === field;

        return (
            <div
                className={classnames('form__field', {
                    form__field_filled: text || this.isLoading,
                    field__error: notValid,
                    field__valid: passwordConfirmState === 'valid'
                })}
            >
                <Input
                    id={field}
                    value={text}
                    name={field}
                    type={fieldType}
                    autoComplete='new-password'
                    onFocus={this.checkInvalid}
                    onBlur={this.validate}
                    onChange={this.handleInput}
                    view='one-border'
                    state={notValid ? 'error' : ''}
                    filled={!!text || this.isLoading}
                />
                <label htmlFor='password_confirm' className='registration__label'>
                    {i18n('_AUTH_.field_label_repeat-passwd2')}
                </label>

                {!hintActive && isActive && !this.state.forceHidden && (
                    <FieldPopup fieldName='password_confirm'>
                        <div className='form__popup-error' data-t='password_confirm-error' role='alert'>
                            <div className='error-message'>{passwordConfirmError.text}</div>
                        </div>
                    </FieldPopup>
                )}
            </div>
        );
    }
}

RegistrationPasswordConfirm.propTypes = {
    hintActive: PropTypes.bool,
    value: PropTypes.string.isRequired,
    passwordValue: PropTypes.string,
    passwordConfirmState: PropTypes.string.isRequired,
    passwordConfirmError: PropTypes.object,
    fieldType: PropTypes.string.isRequired,
    updateField: PropTypes.func.isRequired,
    activeField: PropTypes.string,
    defineActiveField: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired
};
