import './password_input.styl';
import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Input} from '@components/Input';
import FieldPopup from '@blocks/registration/desktop/errorPopup/error_popup.jsx';
import classnames from 'classnames';
import checkIfInvalid from '@blocks/registration/methods/checkIfInvalid';
import validatePassword from '@blocks/registration/methods/validatePassword';
import {updateHintStatus} from '@blocks/actions/form';
import debounce from 'lodash/debounce';
import metrics from '@blocks/metrics';

const PasswordProgressBar = (props) => {
    const indicatorStyle = {
        width: `${props.indicatorStrength}%`,
        backgroundColor: props.color
    };

    return (
        <span className='progress-base'>
            <span className='progress-indicator' style={indicatorStyle} />
        </span>
    );
};

PasswordProgressBar.propTypes = {
    indicatorStrength: PropTypes.number,
    color: PropTypes.string
};

const PopupContent = (props) => (
    <div
        className={classnames('form__popup-error', {password__popup: props.errorTextDescription})}
        data-t='password-error'
        role='alert'
    >
        <div
            className={classnames('suggest__status-text', {
                'suggest__status-valid': props.status === 'valid',
                'warning-text': props.isWarning
            })}
        >
            {props.errorText}
        </div>
        {props.showProgressBar && (
            <PasswordProgressBar value={props.text} indicatorStrength={props.indicatorStrength} color={props.color} />
        )}
        <span className='suggest__description-text'>
            {props.status === 'valid' ? i18n('Frontend.password_valid_description') : props.errorTextDescription}
        </span>
    </div>
);

PopupContent.propTypes = {
    errorText: PropTypes.string,
    errorTextDescription: PropTypes.string,
    text: PropTypes.string,
    color: PropTypes.string,
    status: PropTypes.string,
    showProgressBar: PropTypes.bool,
    indicatorStrength: PropTypes.number,
    isWarning: PropTypes.bool
};

export default class RegistrationPasswordInput extends PureComponent {
    constructor(props) {
        super(props);

        this.state = {
            indicatorStrength: 0
        };

        this.handleInput = this.handleInput.bind(this);
        this.handleFocusout = this.handleFocusout.bind(this);
        this.handleValidation = debounce(this.handleValidation.bind(this), 600);
        this.definePasswordStrength = this.definePasswordStrength.bind(this);
        this.defineColor = this.defineColor.bind(this);
        this.checkInvalid = this.checkInvalid.bind(this);
        this.checkInactiveFieldPopupVisible = this.checkInactiveFieldPopupVisible.bind(this);
        this.checkActiveFieldPopupVisible = this.checkActiveFieldPopupVisible.bind(this);

        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        if (this.props.passwordState === 'need_validation') {
            this.handleValidation(this.props.value);
        }

        this.forceUpdate();
    }

    handleInput(event) {
        const {target = {}} = event;
        const {value} = target;

        this.props.updateField(event);
        this.handleValidation(value);
    }

    handleValidation(value) {
        const {dispatch} = this.props;

        dispatch(validatePassword(value));
        this.definePasswordStrength(value);
    }

    handleFocusout() {
        this.props.defineActiveField('password_confirm');
        this.props.dispatch(updateHintStatus(false));
        metrics.send(['Ввод пароля']);
    }

    checkInvalid() {
        this.props.dispatch(updateHintStatus(true));
        this.props.defineActiveField('password');
        this.props.dispatch(checkIfInvalid('password'));
    }

    definePasswordStrength(pwd) {
        let percents = pwd.length * 5;
        const hasDigital = /\d/.test(pwd);
        const hasUppercase = /[A-Z]/.test(pwd);
        // eslint-disable-next-line no-useless-escape
        const hasSpecialSymbols = /[-\[\]_.+@#$%^&*]/.test(pwd);

        if (percents > 70) {
            percents = 70;
        }

        if (hasDigital) {
            percents += 10;
        }

        if (hasUppercase) {
            percents += 10;
        }

        if (hasSpecialSymbols) {
            percents += 10;
        }

        this.setState({
            indicatorStrength: percents
        });
    }

    defineColor(status, value) {
        let color = 'red';

        if (value.length >= 6 && status === 'not_valid') {
            color = 'orange';
        } else if (status === 'valid') {
            color = '#37B24D';
        }

        return color;
    }

    checkInactiveFieldPopupVisible() {
        const {isPasswordErrorActive, passwordError, isMobile, passwordState, isPasswordFieldActive} = this.props;
        const isNotValid = passwordState === 'not_valid';
        const isWarning = isNotValid && passwordError.warning;

        if (isMobile) {
            return !isPasswordFieldActive && isNotValid && !isWarning;
        }
        return !isPasswordFieldActive && isNotValid && isPasswordErrorActive && !isWarning;
    }

    checkActiveFieldPopupVisible() {
        const {isPasswordFieldActive, passwordState, passwordError} = this.props;

        return isPasswordFieldActive && passwordState !== '' && passwordError.code !== 'missingvalue';
    }

    render() {
        const field = 'password';
        const {
            value,
            loginValue,
            passwordState,
            fieldType,
            isPasswordFieldActive,
            passwordError,
            toggleFieldType,
            isAutoFocusEnabled
        } = this.props;
        const text = value;
        const isNotValid = passwordState === 'not_valid';
        const isWarning = isNotValid && passwordError.warning;
        const errorText = passwordError.text ? passwordError.text : i18n('_AUTH_.password_valid');
        const errorTextDescription = passwordError.errorDescription ? passwordError.errorDescription : '';
        const color = this.defineColor(passwordState, text);
        const showProgressBar =
            isPasswordFieldActive && text.trim().length > 1 && text.toLowerCase() !== loginValue.toLowerCase();
        const isActiveFieldPopupVisible = this.checkActiveFieldPopupVisible();
        const isInactivePopupVisible = this.checkInactiveFieldPopupVisible();
        const fieldTogglerText = fieldType === 'password' ? i18n('_AUTH_.passwd.view') : i18n('_AUTH_.passwd.hide');

        return (
            <div
                className={classnames('form__field', 'field__password', {
                    form__field_filled: text || this.isLoading,
                    field__error: isNotValid && !isWarning,
                    field__warning: isNotValid && isWarning && !isPasswordFieldActive,
                    field__valid: passwordState === 'valid' && !isPasswordFieldActive
                })}
            >
                {!this.isLoading && (
                    <button
                        className={`field-type__toggler field-type-${fieldType}`}
                        onClick={toggleFieldType}
                        title={fieldTogglerText}
                        tabIndex='2'
                        type='button'
                    >
                        {fieldTogglerText}
                    </button>
                )}

                <Input
                    id={field}
                    value={text}
                    name={field}
                    type={fieldType}
                    autoFocus={isAutoFocusEnabled}
                    autoComplete='new-password'
                    onBlur={this.handleFocusout}
                    onFocus={this.checkInvalid}
                    onChange={this.handleInput}
                    view='one-border'
                    state={isNotValid && !isWarning ? 'error' : ''}
                    filled={!!text || this.isLoading}
                />
                <label htmlFor={field} className='registration__label'>
                    {i18n('_AUTH_.field_password')}
                </label>
                {text.trim().length > 1 && (
                    <FieldPopup visible={isActiveFieldPopupVisible}>
                        <PopupContent
                            text={text}
                            showProgressBar={showProgressBar}
                            indicatorStrength={this.state.indicatorStrength}
                            color={color}
                            status={passwordState}
                            errorText={errorText}
                            isWarning={isWarning}
                            errorTextDescription={errorTextDescription}
                        />
                    </FieldPopup>
                )}
                {isInactivePopupVisible && (
                    <FieldPopup fieldName='password' visible={isInactivePopupVisible}>
                        <div className='form__popup-error' data-t='password-inactive-error' role='alert'>
                            <div className='error-message'>{errorText}</div>
                        </div>
                    </FieldPopup>
                )}
            </div>
        );
    }
}

RegistrationPasswordInput.propTypes = {
    value: PropTypes.string.isRequired,
    loginValue: PropTypes.string,
    passwordState: PropTypes.string.isRequired,
    isPasswordErrorActive: PropTypes.bool,
    isPasswordFieldActive: PropTypes.bool,
    passwordError: PropTypes.object.isRequired,
    fieldType: PropTypes.string.isRequired,
    updateField: PropTypes.func.isRequired,
    toggleFieldType: PropTypes.func.isRequired,
    defineActiveField: PropTypes.func.isRequired,
    dispatch: PropTypes.func.isRequired,
    isAutoFocusEnabled: PropTypes.bool,
    isMobile: PropTypes.bool
};
