import api from '@blocks/api';
import {
    MAGIC_CALL_CONFIRMATION_METRIC_PARAM,
    DICTATION_CALL_CONFIRMATION_METRIC_PARAM,
    DISCARD_CALL_CONFIRMATION_METRIC_PARAM,
    SMS_CONFIRMATION_METRIC_PARAM,
    SUCCESS_CONFIRMATION_CALL_METRIC_PARAM,
    SUCCESS_CONFIRMATION_SMS_METRIC_PARAM
} from '@blocks/registration/actions';
import {
    updateHumanConfirmationStatus,
    updateConfirmationFetchingStatus,
    setCallConfirmationTimer
} from '@blocks/actions/form';
import updateFieldStatus from './updateFieldStatus';
import checkIfFieldEmpty from './checkIfFieldEmpty';
import {errorsTxt} from '@blocks/registration/errors.js';
import metrics from '@blocks/metrics';

export default function checkPhoneCode(code) {
    return (dispatch, getState) => {
        const {
            common: {track_id: trackId},
            form: {
                isPhoneCallConfirmationAvailable,
                isDiscardCallConfirmation,
                validation: {phoneConfirmationType, callConfirmationTimer}
            }
        } = getState();
        const errors = errorsTxt.phone.errors;
        const errorObj = {
            code: 'code.missingvalue',
            text: errors['code.missingvalue']
        };
        const isCallConfirmation = phoneConfirmationType === 'call';

        dispatch(updateConfirmationFetchingStatus({isFetching: true}));

        if (checkIfFieldEmpty(code, 'phoneCode')(dispatch, errorObj)) {
            dispatch(updateConfirmationFetchingStatus({isFetching: false}));
            return;
        }

        api.request('phone-confirm-code', {track_id: trackId, code})
            .then(() => {
                updateFieldStatus('phoneCode', 'valid')(dispatch, getState);
                dispatch(updateHumanConfirmationStatus(true));

                if (callConfirmationTimer) {
                    clearTimeout(callConfirmationTimer);
                    dispatch(setCallConfirmationTimer(null));
                }

                sendMetrics({
                    isCallConfirmation,
                    isPhoneCallConfirmationAvailable,
                    isDiscardCallConfirmation
                });
            })
            .fail((error) => {
                const errorCodes = error.errors ? error.errors : [];

                errorObj.code = error.errors ? error.errors[0] : '';
                if (errorCodes.length > 0) {
                    const errorTextArr = errorCodes.map((err) => errors[err]);

                    errorObj.text = errorTextArr[0];
                }
                updateFieldStatus('phoneCode', 'not_valid')(dispatch, errorObj);
            })
            .always(() => {
                dispatch(updateConfirmationFetchingStatus({isFetching: false}));
            });
    };
}

const METRIC_PARAMS = {
    sms: [SMS_CONFIRMATION_METRIC_PARAM, SUCCESS_CONFIRMATION_SMS_METRIC_PARAM],
    sms_for_magic: [MAGIC_CALL_CONFIRMATION_METRIC_PARAM, SUCCESS_CONFIRMATION_SMS_METRIC_PARAM],
    sms_for_dictation: [DICTATION_CALL_CONFIRMATION_METRIC_PARAM, SUCCESS_CONFIRMATION_SMS_METRIC_PARAM],
    sms_for_discard_call: [DISCARD_CALL_CONFIRMATION_METRIC_PARAM, SUCCESS_CONFIRMATION_SMS_METRIC_PARAM],
    call_for_dictation: [DICTATION_CALL_CONFIRMATION_METRIC_PARAM, SUCCESS_CONFIRMATION_CALL_METRIC_PARAM],
    discard_call: [DISCARD_CALL_CONFIRMATION_METRIC_PARAM, SUCCESS_CONFIRMATION_CALL_METRIC_PARAM]
};

function sendMetrics(options) {
    const {isCallConfirmation, isPhoneCallConfirmationAvailable, isDiscardCallConfirmation} = options;

    let metricParams = [];

    let metricGoalTarget = 'registration_success_confirmation';

    let confirmationType = 'sms';

    if (isCallConfirmation) {
        confirmationType = isDiscardCallConfirmation ? 'discard_call' : 'call_for_dictation';
    } else if (isPhoneCallConfirmationAvailable) {
        if (isDiscardCallConfirmation) {
            confirmationType += '_for_discard_call';
        } else {
            confirmationType += '_for_dictation';
        }
    }

    metricGoalTarget += `_${confirmationType}`;
    metricParams = METRIC_PARAMS[confirmationType] || METRIC_PARAMS.sms;

    metrics.send(metricParams);
    metrics.goal(metricGoalTarget);
}
