import api from '../../api';
import {
    updateRegistrationErrors,
    updateFetchingStatus,
    REGISTRATION_PDD_GOAL_PREFIX,
    REGISTRATION_PDD_MOBILE_GOAL_PREFIX,
    REGISTRATION_MOBILE_GOAL_PREFIX,
    REGISTRATION_GOAL_PREFIX
} from '../actions';
import returnFormData from './returnFormData';
import updateFieldStatus from './updateFieldStatus';
import sendConnectOrCompleteRegistrationData from './sendConnectOrCompleteRegistrationData';
import {sendChildishRegistrationData} from './sendChildishRegistrationData';
import metrics from '../../metrics';
import {errorsTxt} from '../errors.js';

export const defineRegistrationFunction = (prefix = '', registrationType) => {
    const isConnectOrCompleteReg =
        prefix === REGISTRATION_PDD_MOBILE_GOAL_PREFIX ||
        prefix === REGISTRATION_PDD_GOAL_PREFIX ||
        prefix.indexOf('complete') !== -1;

    if (registrationType === 'childish') {
        return sendChildishRegistrationData;
    }

    if (isConnectOrCompleteReg) {
        return sendConnectOrCompleteRegistrationData;
    }

    return sendRegistrationData;
};

export const sendRegistrationData = (data) => {
    return (dispatch, getState) => {
        const {
            settings: {uatraits},
            common: {track_id: trackId, csrf, currentPage, retpath},
            additionalStep = {},
            form: {type: formType}
        } = getState();
        const isMobile = uatraits.isMobile && !uatraits.isTablet;
        const baseRedirectUrl = `/registration/finish?track_id=${trackId}`;
        const params = data || dispatch(returnFormData());
        const metricsPrefix = isMobile ? REGISTRATION_MOBILE_GOAL_PREFIX : REGISTRATION_GOAL_PREFIX;

        params.csrf_token = csrf;
        params.track_id = trackId;
        params.eula_accepted = 'on';

        if (formType) {
            params.type = formType;
        }

        api.request('registration-alternative', params)
            .then((response) => {
                if (response.status === 'ok') {
                    metrics.send(['Форма', 'Успешная регистрация']);
                    metrics.goal(`${metricsPrefix}_form_success`);

                    if (additionalStep.type && ['avatar', 'qr'].includes(additionalStep.type)) {
                        location.href = additionalStep.path || retpath;
                        return;
                    }

                    location.href = response.target || baseRedirectUrl;
                    return;
                }

                dispatch(
                    updateRegistrationErrors({
                        status: 'error',
                        code: 'global',
                        text: errorsTxt.global.errors.smthWrong,
                        descriptionText: ''
                    })
                );
            })
            .fail((error) => {
                dispatch(updateFetchingStatus(false));

                if (error && Array.isArray(error.error)) {
                    const errorList = error.error;
                    const errorCode = errorList[0] && errorList[0].code;
                    const isFieldError = errorList.find((item) => item.field);
                    const regError = {
                        status: 'error',
                        code: errorCode || 'global',
                        text: errorsTxt.global.errors.global,
                        descriptionText: ''
                    };

                    if (errorCode === 'registrationsmssendperiplimitexceeded') {
                        regError.text = errorsTxt.registration.errors.registrationsmssendperiplimitexceeded;
                        dispatch(updateRegistrationErrors(regError));
                        return;
                    }

                    if (errorCode === 'sessionidoverflow') {
                        regError.text = errorsTxt.registration.errors.sessionidoverflow;
                        regError.descriptionText = errorsTxt.registration.descriptions.sessionidoverflow;

                        dispatch(updateRegistrationErrors(regError));
                        return;
                    }
                    if (errorCode === 'registrationalreadycompletederror') {
                        // PASSP-17624
                        window.location.href = currentPage;
                    }

                    metrics.send(['Форма', 'Показ ошибок', errorCode]);
                    metrics.goal(`${metricsPrefix}_form_fail`);

                    if (errorList && isFieldError && errorCode) {
                        errorList.forEach((errorItem) => {
                            updateFieldStatus(errorItem.field, 'not_valid')(dispatch, {
                                code: errorItem.code,
                                text: errorsTxt[errorItem.field].errors
                                    ? errorsTxt[errorItem.field].errors[errorItem.code]
                                    : errorsTxt.global.errors.global
                            });
                        });
                        return;
                    }

                    dispatch(updateRegistrationErrors(regError));
                    return;
                }

                dispatch(
                    updateRegistrationErrors({
                        status: 'error',
                        code: 'global',
                        text: errorsTxt.global.errors.smthWrong,
                        descriptionText: ''
                    })
                );
            });
    };
};
