import api from '@blocks/api';
import metrics from '@blocks/metrics';
import {updateRegistrationErrors, updateFetchingStatus} from '@blocks/registration/actions';
import {setCaptchaRequired} from '@blocks/actions/form';
import updateFieldStatus from './updateFieldStatus';
import {errorsTxt} from '@blocks/registration/errors.js';

export default function sendConnectOrCompleteRegistrationData(data) {
    return (dispatch, getState) => {
        const {
            common: {csrf, track_id: trackId, retpath},
            form: {prefix}
        } = getState();
        const baseRedirectUrl = `/registration/finish?track_id=${trackId}`;
        const params = data;

        params.csrf_token = csrf;
        params.track_id = trackId;
        params.eula_accepted = 'on';

        if (retpath) {
            params.retpath = retpath;
        }

        return api
            .request(`registration-${prefix.indexOf('complete') !== -1 ? 'complete' : 'connect'}`, params)
            .then((response) => {
                if (response.status === 'ok') {
                    window.location.href = response.target || baseRedirectUrl;

                    metrics.send(['Форма', 'Успешная регистрация']);
                    metrics.goal(`${prefix}_form_success`);
                }
            })
            .fail((errors) => {
                dispatch(updateFetchingStatus(false));

                if (!errors || !Array.isArray(errors.error) || !errors.error[0]) {
                    const regError = {
                        status: 'global',
                        code: 'global',
                        text: errorsTxt.global.errors.smthWrong
                    };

                    dispatch(updateRegistrationErrors(regError));
                    return;
                }

                const error = errors.error[0];

                let field;

                let errorCode;

                let codeTail;

                if (error === 'captcha.required') {
                    dispatch(
                        updateRegistrationErrors({
                            status: '',
                            code: '',
                            text: ''
                        })
                    );
                    dispatch(setCaptchaRequired(true));
                    return;
                }

                if (typeof error === 'string') {
                    const parts = error.split('.');

                    field = parts[0];
                    codeTail = parts[1];
                    errorCode = error;
                } else {
                    field = error.field;
                    errorCode = error.code;
                    codeTail = errorCode;
                }

                if (errorCode === 'domain.invalid_type') {
                    errorCode = 'domain.invalid';
                }

                if (
                    errorCode === 'sessionidoverflow' ||
                    error === 'sessionid.overflow' ||
                    errorCode === 'sessionid.overflow'
                ) {
                    const regError = {
                        status: 'error',
                        code: 'sessionidoverflow',
                        text: errorsTxt.registration.errors.sessionidoverflow,
                        descriptionText: errorsTxt.registration.descriptions.sessionidoverflow
                    };

                    dispatch(updateRegistrationErrors(regError));
                    return;
                }

                if (errorCode === 'registrationsmssendperiplimitexceeded') {
                    const regError = {
                        status: 'error',
                        code: 'registrationsmssendperiplimitexceeded',
                        text: errorsTxt.registration.errors.registrationsmssendperiplimitexceeded
                    };

                    dispatch(updateRegistrationErrors(regError));
                    return;
                }

                if (!errorsTxt[field] || (!errorsTxt[field].errors[errorCode] && !errorsTxt[field].errors[codeTail])) {
                    const regError = {
                        status: 'global',
                        code: 'global',
                        text: errorsTxt.global.errors.smthWrong
                    };

                    dispatch(updateRegistrationErrors(regError));
                    return;
                }

                updateFieldStatus(field, 'not_valid')(dispatch, {
                    code: errorCode,
                    text: errorsTxt[field].errors[errorCode] || errorsTxt[field].errors[codeTail]
                });
                metrics.send(['Форма', 'Показ ошибок', errorCode]);
                metrics.goal(`${prefix}_form_fail`);
            });
    };
}
