import api from '@blocks/api';
import {
    updateStates,
    updateValues,
    updateHumanConfirmationStatus,
    updateConfirmationFetchingStatus,
    changePhoneConfirmationType,
    setCallConfirmationTimer,
    setConfirmationCodeLength,
    setCaptchaRequired
} from '@blocks/actions/form';
import {updatePhoneConfirmationCallNumber} from '@blocks/registration/actions';
import updateFieldStatus from './updateFieldStatus';
import findFieldsWithErrors from './findFieldsWithErrors';
import checkIfFieldEmpty from './checkIfFieldEmpty';
import checkPhoneConfirmationStatus from './checkPhoneConfirmationStatus';
import {errorsTxt} from '@blocks/registration/errors';
import sendMetricsForSendingConfirmationCode from './sendMetricsForSendingConfirmationCode';
import {isAndroidAm} from '@blocks/selectors';

const RETRY_WITH_SMS_STATUSES = ['calls.shut_down', 'create_call.failed', 'exception.unhandled'];

export default function submitPhoneConfirmationCode(phone, isCallConfirmation = false, sendAgain = false) {
    return (dispatch, getState) => {
        const state = getState();
        const {
            common: {track_id: trackId},
            form: {
                isPhoneCallConfirmationAvailable,
                isDiscardCallConfirmation,
                validation: {isValidPhoneForCall, callConfirmationTimer}
            }
        } = state;
        const errors = errorsTxt.phone.errors;
        const requestParams = {
            track_id: trackId,
            number: phone
        };

        if (!isCallConfirmation && !isAndroidAm(state)) {
            requestParams.isCodeWithFormat = true;
        }

        if (callConfirmationTimer) {
            clearTimeout(callConfirmationTimer);
            dispatch(setCallConfirmationTimer(null));
        }

        dispatch(updateConfirmationFetchingStatus({isFetching: true}));

        if (checkIfFieldEmpty(phone, 'phone')(dispatch, {code: 'missingvalue', text: errors.missingvalue})) {
            dispatch(updateConfirmationFetchingStatus({isFetching: false}));
            return;
        }

        dispatch(updateStates({field: 'phoneCode', status: ''}));
        dispatch(updateStates({field: 'phone', status: ''}));
        dispatch(updateValues({field: 'phoneCode', value: ''}));

        const isCallConfirmationMethod = isCallConfirmation && isPhoneCallConfirmationAvailable && isValidPhoneForCall;

        if (isCallConfirmationMethod) {
            requestParams.confirm_method = 'by_call';
            dispatch(changePhoneConfirmationType('call'));
        } else {
            requestParams.confirm_method = 'by_sms';
            dispatch(changePhoneConfirmationType('sms'));
        }

        if (isDiscardCallConfirmation && isCallConfirmationMethod) {
            requestParams.confirm_method = 'by_call';
        }

        api.request('phone-confirm-code-submit', requestParams)
            .done((response) => {
                if (response.status === 'ok') {
                    dispatch(updateStates({field: 'phoneCodeStatus', status: 'code_sent'}));
                    updateFieldStatus('phone', 'valid')(dispatch, getState);
                    dispatch(findFieldsWithErrors());

                    if (response.code_length) {
                        dispatch(setConfirmationCodeLength(response.code_length));
                    }

                    if (['by_call', 'by_flash_call'].indexOf(requestParams.confirm_method) > -1) {
                        const timer = setTimeout(() => {
                            dispatch(checkPhoneConfirmationStatus());
                        }, 5000);

                        dispatch(setCallConfirmationTimer(timer));

                        if (response.calling_number_template) {
                            dispatch(updatePhoneConfirmationCallNumber(response.calling_number_template));
                        }
                    }
                    dispatch(sendMetricsForSendingConfirmationCode(sendAgain, isCallConfirmation));
                }
            })
            .fail((error) => {
                let errorCode = error.errors[0];

                let errorText = errors[errorCode] || '';

                if (RETRY_WITH_SMS_STATUSES.indexOf(errorCode) > -1) {
                    dispatch(changePhoneConfirmationType('sms'));
                    dispatch(submitPhoneConfirmationCode(phone, false));
                    return;
                }

                if (errorCode === 'captcha.required') {
                    return dispatch(setCaptchaRequired(true));
                }

                if (!errorText && requestParams.confirm_method === 'by_call') {
                    errorCode = 'call_confirm.failed';
                    errorText = errors[errorCode];
                }

                if (errorCode === 'phone.confirmed') {
                    updateFieldStatus('phone', 'valid')(dispatch, getState);
                    updateFieldStatus('phoneCode', 'valid')(dispatch, getState);
                    dispatch(updateHumanConfirmationStatus(true));
                    return;
                }

                if (errorCode === 'sms_limit.exceeded') {
                    dispatch(updateStates({field: 'phoneCodeStatus', status: ''}));
                }

                updateFieldStatus('phone', 'not_valid')(dispatch, {code: errorCode, text: errorText});
            })
            .always(() => {
                dispatch(updateConfirmationFetchingStatus({isFetching: false}));
            });
    };
}
