import {errorsTxt as commonErrorsTxt} from '@blocks/registration/errors';
import {errorsTxt} from '@blocks/registration-lite/errors';
import {showEulaPopup} from '@blocks/registration/actions';
import {updateErrors, updateQuestionCustomState} from '@blocks/actions/form';
import handleValidationError from './handleValidationError';
import sendDataForRegistration from './sendDataForRegistration';
import regMethods from './basicRegistrationMethods';
import metrics from '@blocks/metrics';
import processPhoneData from './phoneConfirmation/processPhoneData';

const allErrorTexts = Object.assign({}, commonErrorsTxt, errorsTxt);

export default function validateForm() {
    return (dispatch, getState) => {
        const state = getState();
        const {form, registrationType, registrationName, settings, person = {}} = state;
        const {states, values, validation, errors, isEulaShowedInPopup, type} = form;
        const {captchaRequired} = form;
        const {hasRecoveryMethod, isLiteWithoutPassword} = person;
        const method = validation.method;
        const isMobile = settings.uatraits.isMobile && !settings.uatraits.isTablet;
        const updatedStates = Object.assign({}, states);
        const updatedValues = getFormValues(state);
        const validatedStates = regMethods.prepareFormData(updatedStates, method, isMobile, registrationType);
        const emptyFields = {};
        const errorFields = {};

        let isValid = true;

        if (!validation.humanConfirmationDone && method === 'phone') {
            if (!updatedValues.phoneCode && updatedValues.password && updatedValues.login) {
                return dispatch(processPhoneData());
            }
            isValid = false;
        }

        if (!captchaRequired && hasRecoveryMethod && validatedStates.hasOwnProperty('captcha')) {
            delete validatedStates.captcha;
        }

        if (method === 'captcha') {
            const CUSTOM_QUESTION_ID = '99';

            if (updatedValues.hint_question_id === CUSTOM_QUESTION_ID && !updatedValues.hint_question_custom) {
                validatedStates.hint_question_custom = 'not_valid';
                handleQuestionError(dispatch);
            }
            if (updatedValues.hint_question_id === CUSTOM_QUESTION_ID) {
                updatedValues.hint_question = values.hint_question_custom;
            }
        }

        for (const fieldName in validatedStates) {
            const hasWarningStatus = fieldName === 'password' && errors[fieldName].warning;
            const isErrorTextAvailable = allErrorTexts[fieldName] !== undefined;

            if (validatedStates[fieldName] === 'valid' || hasWarningStatus) {
                continue;
            }

            isValid = false;
            emptyFields[fieldName] = 'not_valid';

            if (isErrorTextAvailable) {
                let errorCode = errors[fieldName].code || 'missingvalue';
                const isEmptyPhone = fieldName === 'phone' && errorCode === 'missingvalue';
                const errorTextDescription = allErrorTexts[fieldName].descriptions;

                if (
                    type === 'complete_lite' &&
                    !isLiteWithoutPassword &&
                    (errorCode === 'empty' || errorCode === 'missingvalue') &&
                    fieldName === 'password'
                ) {
                    errorCode += '_current';
                }

                errorFields[fieldName] = {
                    code: errorCode,
                    text: allErrorTexts[fieldName].errors[errorCode],
                    descriptionText:
                        errorTextDescription && errorTextDescription[errorCode] ? errorTextDescription[errorCode] : ''
                };

                if (isEmptyPhone) {
                    errorFields[fieldName].text = allErrorTexts.phone.errors.needsconfirmation;
                }
            }
        }

        if (!isValid) {
            return handleValidationError(dispatch, emptyFields, errorFields, isMobile);
        }

        updatedValues['human-confirmation'] = method;

        if (
            (isEulaShowedInPopup && registrationName !== 'complete') ||
            (isEulaShowedInPopup && checkIfRegCompleteWithEula(type, registrationName))
        ) {
            metrics.send(['Форма', 'Показ попапа с Eula']);
            return dispatch(showEulaPopup(true));
        }
        sendDataForRegistration(dispatch, updatedValues);
    };
}

function handleQuestionError(dispatch) {
    const hintQuestionErrorObj = {
        code: 'missingvalue',
        text: allErrorTexts.hint.hintQuestionErrors.missingvalue
    };

    dispatch(updateQuestionCustomState({status: 'not_valid'}));
    dispatch(updateErrors({field: 'hint_question_custom', error: hintQuestionErrorObj, active: true}));
}

function checkIfRegCompleteWithEula(type, registrationName) {
    const isRegistrationWithEula = type && !['complete_social', 'complete_social_with_login'].includes(type);
    const isCompleteRegistration = registrationName === 'complete';

    return isCompleteRegistration && isRegistrationWithEula;
}

export function getFormValues(state) {
    const {form, registrationName, person} = state;
    const {values, type, captchaRequired} = form;
    const isCompleteRegistration = registrationName === 'complete';
    const formValues = Object.assign({}, values);

    if (isCompleteRegistration) {
        const {hasRecoveryMethod} = person;

        if (!captchaRequired && hasRecoveryMethod && values.hasOwnProperty('captcha')) {
            delete formValues.captcha;
        }

        formValues.state = type;
    }
    return formValues;
}
