import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';
import debounce from 'lodash/debounce';

import {Input} from '@components/Input';
import {Spin} from '@components/Spin';

import LoginSuggestField from './login_suggest_field.jsx';
import LoginSuggestRow from './login_suggest_row.jsx';

import * as extracted from './login_field.js';

export class LoginField extends PureComponent {
    static mapStateToProps(state) {
        const {form, logins} = state;
        const {values, states, errors} = form;

        return {
            activeError: errors.active,
            fieldError: errors.login,
            fieldState: states.login,
            fieldValue: values.login,
            firstname: values.firstname,
            lastname: values.lastname,
            logins
        };
    }

    constructor(props) {
        super(props);

        this.state = {
            isActive: false,
            isValid: false
        };

        this.handleInput = extracted.handleInput.bind(this);
        this.handleFocus = extracted.handleFocus.bind(this);
        this.handleFocusout = extracted.handleFocusout.bind(this);
        this.handleValidation = debounce(extracted.handleValidation.bind(this), 600);

        this.timeOut = 2000;
        this.loginFieldFocusTimeout = null;
        this.loginFieldFocusOutTimeout = null;

        this.isLoading = true;
    }

    componentDidMount() {
        this.isLoading = false;

        if (this.props.fieldState === 'need_validation') {
            this.handleValidation(this.props.fieldValue);
        }

        this.forceUpdate();
    }

    componentWillUnmount() {
        extracted.clearAll.call(this);
    }

    render() {
        const {fieldState, fieldValue, fieldError, activeError, logins, isAutoComplete, dispatch} = this.props;
        const isNotValid = fieldState === 'not_valid' || activeError === 'login';
        const isValid = fieldState === 'valid';

        return (
            <div className='field-wrapper login__field-wrapper'>
                <div
                    className={classnames('form__field form__field_wide', {
                        form__field_filled: fieldValue || this.isLoading,
                        field__error: isNotValid,
                        field__valid: isValid
                    })}
                >
                    <div className='validating-throbber'>
                        <Spin progress={logins.isValidating} size='xs' />
                    </div>
                    {isAutoComplete === false && (
                        <div className='registration__chrome_bag_autocomplete'>
                            <input type='input' name='login' tabIndex='-1' />
                        </div>
                    )}
                    <Input
                        id='login'
                        name='login'
                        autoComplete='off'
                        value={fieldValue}
                        ref={(field) => {
                            this.loginField = field;
                        }}
                        onChange={this.handleInput}
                        onFocus={this.handleFocus}
                        onBlur={this.handleFocusout}
                        view='one-border'
                        state={isNotValid ? 'error' : ''}
                        filled={Boolean(fieldValue) || this.isLoading}
                    />
                    <label htmlFor='login' className='registration__label'>
                        {i18n('Profile.personal.data.login')}
                    </label>
                </div>
                {(
                    <LoginSuggestRow
                        logins={logins.loginsList}
                        error={fieldError || {}}
                        dispatch={dispatch}
                        isValid={isValid}
                    />
                ) || (
                    <LoginSuggestField
                        logins={logins}
                        loginError={fieldError}
                        loginState={fieldState}
                        activeError={activeError}
                        fieldActive={this.state.isActive}
                        dispatch={dispatch}
                    />
                )}
            </div>
        );
    }
}

LoginField.propTypes = {
    fieldState: PropTypes.string.isRequired,
    fieldValue: PropTypes.string.isRequired,
    fieldError: PropTypes.oneOfType([PropTypes.string, PropTypes.object]).isRequired,
    logins: PropTypes.object.isRequired,
    firstname: PropTypes.string.isRequired,
    isAutoComplete: PropTypes.bool,
    lastname: PropTypes.string.isRequired,
    activeError: PropTypes.string.isRequired,
    dispatch: PropTypes.func.isRequired
};

export default connect(LoginField.mapStateToProps)(LoginField);
