import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {Button} from '@components/Button';
import {Spin} from '@components/Spin';
import metrics from '@blocks/metrics';
import {REGISTRATION_MOBILE_GOAL_PREFIX} from '@blocks/registration/actions';
import {updateValues, updateStates, updateErrorsValid} from '@blocks/actions/form';
import LoginSuggestOption from './login_suggest_option.jsx';

export default class LoginSuggestField extends PureComponent {
    constructor(props) {
        super(props);

        this.setSelectedLogin = this.setSelectedLogin.bind(this);
        this.showMoreLogins = this.showMoreLogins.bind(this);
        this.addLoginsToSuggest = this.addLoginsToSuggest.bind(this);

        this.state = {
            showList: false,
            showSuggestedLogins: false,
            loginFromSuggestPicked: false,
            showAll: false,
            loginList: []
        };

        this.addLoginListInterval = null;
    }

    componentWillUpdate(nextProps) {
        if (nextProps.logins.loginsList.length !== this.props.logins.loginsList.length) {
            this.setState(() => ({showAll: false, loginList: []}));

            const {loginsList} = nextProps.logins;
            const loginsToShow = loginsList.slice(0, 3);

            this.addLoginsToSuggest(loginsToShow);
        }
    }

    componentWillUnmount() {
        if (this.addLoginListInterval) {
            clearInterval(this.addLoginListInterval);
        }
    }

    addLoginsToSuggest(logins) {
        const self = this;

        if (this.addLoginListInterval) {
            clearInterval(this.addLoginListInterval);
        }

        this.addLoginListInterval = setInterval(() => {
            if (!logins.length) {
                clearInterval(this.addLoginListInterval);
                return;
            }

            self.setState((prevState) => ({
                loginList: prevState.loginList.concat(logins.splice(0, 1))
            }));
        }, 100);
    }

    setSelectedLogin(event) {
        const target = event.target;
        const selectedLogin = target.id;
        const {dispatch} = this.props;

        dispatch(updateValues({field: 'login', value: selectedLogin}));
        dispatch(updateStates({field: 'login', status: 'valid'}));
        dispatch(updateErrorsValid('login'));

        metrics.send(['Саджест логина', 'Выбор логина из списка']);
        metrics.goal(`${REGISTRATION_MOBILE_GOAL_PREFIX}_suggest_selected`);

        this.setState({
            showSuggestedLogins: false,
            loginFromSuggestPicked: true,
            showAll: false,
            showList: false
        });

        window.scrollTo(0, 0);
    }

    showMoreLogins() {
        this.setState({
            showAll: true,
            showList: true
        });

        const {loginsList} = this.props.logins;
        const loginsToShow = loginsList.slice(3);

        this.addLoginsToSuggest(loginsToShow);

        metrics.send(['Саджест логина', 'Посмотреть все варианты']);
        metrics.goal(`${REGISTRATION_MOBILE_GOAL_PREFIX}_suggest_more`);
    }

    /* eslint-disable no-extra-parens */
    render() {
        const {activeError, loginError, loginState, logins, fieldActive} = this.props;
        const {loginsList, isFetching} = logins;
        const isVisible = activeError === 'login' || (loginsList.length > 0 && fieldActive) || this.state.showList;
        const loginsToShow = this.state.showAll ? loginsList : loginsList.slice(0, 3);
        const isValidLogin = loginState === 'valid';
        const blockClass = loginsList.length > 0 ? 'form__login-suggest' : 'form__login-message';
        const isShowMoreBtnVisible = !this.state.showAll && loginsList.length > 3;
        const isShowSpinner = loginsToShow.length === 0 && isFetching;

        if (!isVisible || isValidLogin || loginError.code === 'missingvalue') {
            return null;
        }

        return (
            <div className={blockClass}>
                {loginError.text && <div className='form__field-error error-message'>{loginError.text}</div>}

                {loginError.descriptionText && (
                    <div className='suggest__description-text'>{loginError.descriptionText}</div>
                )}

                {this.state.loginList.length !== 0 && (
                    <div className='suggest__logins'>
                        <div className='logins__list-description'>{i18n('Frontend.login_free_list')}:</div>
                        <div className='logins__list'>
                            {this.state.loginList.map((login) => (
                                <LoginSuggestOption
                                    key={login}
                                    login={login}
                                    setSelectedLogin={this.setSelectedLogin}
                                />
                            ))}
                        </div>
                    </div>
                )}

                {isShowSpinner && (
                    <div className='suggest__spinner'>
                        <Spin progress={true} size='m' />
                    </div>
                )}

                {isShowMoreBtnVisible && (
                    <div className='login__suggest-button registration__btn-normal'>
                        <Button view='pseudo' size='l' width='max' onClick={this.showMoreLogins}>
                            {i18n('Frontend.show_all_variants')}
                        </Button>
                    </div>
                )}
            </div>
        );
    }
    /* eslint-enable no-extra-parens */
}

LoginSuggestField.propTypes = {
    logins: PropTypes.object.isRequired,
    loginError: PropTypes.oneOfType([PropTypes.string, PropTypes.object]).isRequired,
    loginState: PropTypes.string.isRequired,
    activeError: PropTypes.string.isRequired,
    fieldActive: PropTypes.bool.isRequired,
    dispatch: PropTypes.func.isRequired
};
