import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import classnames from 'classnames';
import metrics from '@blocks/metrics';
import {clearSuggestedLogins, REGISTRATION_MOBILE_OS_GOAL_PREFIX} from '@blocks/registration/actions';
import {updateValues, updateStates, updateErrorsValid} from '@blocks/actions/form';

export default class LoginSuggestRow extends PureComponent {
    state = {
        faded: false
    };

    componentDidUpdate(prevProps) {
        const {logins} = this.props;

        if (prevProps.logins.length !== logins.length && !this.state.faded) {
            setTimeout(() => {
                this.setState({faded: true});
            }, 300);
        }
    }

    selectLogin = (event) => {
        const loginValue = event.currentTarget && event.currentTarget.dataset.login;
        const {dispatch} = this.props;

        dispatch(updateValues({field: 'login', value: loginValue}));
        dispatch(updateStates({field: 'login', status: 'valid'}));
        dispatch(updateErrorsValid('login'));
        dispatch(clearSuggestedLogins());

        metrics.send(['Саджест логина', 'Выбор логина из списка']);
        metrics.goal(`${REGISTRATION_MOBILE_OS_GOAL_PREFIX}_suggest_selected`);
    };

    /* eslint-disable no-extra-parens */
    render() {
        const {error = {}, logins, isValid} = this.props;

        if ((!logins.length && !error.text) || isValid) {
            return null;
        }

        return (
            <div className={logins.length > 0 ? 'form__login-suggest single__suggest' : 'form__login-message'}>
                {error.text && <div className='form__field-error error-message display_inline-block'>{error.text}</div>}
                {error.descriptionText && error.code !== 'login.not_available' && (
                    <div className='suggest__description-text'>{error.descriptionText}</div>
                )}
                {logins.length !== 0 && !isValid && (
                    <div className='suggest__options'>
                        <div className='logins__list-description'>{i18n('Frontend.login_free_list')}:</div>
                        <div className='list_row'>
                            <div className='list_row__inner'>
                                <div className='logins__list'>
                                    {logins.map((login, ndx) => (
                                        <div
                                            key={ndx}
                                            data-login={login}
                                            className={classnames('row_link', {
                                                suggest__option_fade: this.state.faded
                                            })}
                                            onClick={this.selectLogin}
                                        >
                                            <div className='suggest__option'>{login}</div>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </div>
                    </div>
                )}
            </div>
        );
    }
    /* eslint-enable no-extra-parens */
}

LoginSuggestRow.propTypes = {
    logins: PropTypes.array.isRequired,
    error: PropTypes.shape({
        text: PropTypes.string,
        descriptionText: PropTypes.string,
        code: PropTypes.string
    }).isRequired,
    isValid: PropTypes.bool.isRequired,
    dispatch: PropTypes.func.isRequired
};
