import {createSelector} from 'reselect';
import {TRACK_TYPES} from '@blocks/actions/tracks';
import * as utils from '@blocks/utils';
import {Platforms, RegType} from '@blocks/authv2/nativeMobileApi/constants';

export const getFormValues = createSelector(
    (state = {}) => state.form,
    (form = {}) => form.values || {}
);

export const getFormErrors = createSelector(
    (state = {}) => state.form,
    (form = {}) => form.errors || {}
);

export const getVersion = createSelector(
    (state = {}) => state.settings,
    (settings = {}) => settings.version || 'unknown'
);

export const getMetricsUserType = createSelector(
    (state = {}) => state.settings,
    (settings = {}) => settings.metricsUserType || 'null'
);

export const getUatraits = createSelector(
    (state = {}) => state.settings,
    (settings = {}) => settings.ua || settings.uatraits || {}
);

export const isMobile = (state) => {
    const ua = getUatraits(state);

    return ua.isMobile && !ua.isTablet;
};

export const getLanguage = createSelector(
    (state = {}) => state.settings,
    (settings = {}) => settings.language || settings.lang
);

export const getExperimentsFlags = createSelector(
    (state = {}) => state.common,
    (common = {}) => (common.experiments && common.experiments.flags) || []
);

export const hasExp = (state = {}, expName = '') => getExperimentsFlags(state).includes(expName);

export const getTracks = createSelector(
    (state = {}) => state.tracks,
    (tracks = {}) => tracks
);

export const getTrackId = (state = {}, trackType = '') => {
    const tracks = getTracks(state);

    let track = tracks[`${trackType}TrackId`] || '';

    if (!track && trackType === TRACK_TYPES.COMMON) {
        track = state.common && state.common.track_id;
    }

    return track;
};

export const getAnyTypeTrackId = (state) => {
    const tracks = getTracks(state);
    const trackTypes = Object.keys(tracks);

    return (trackTypes[0] && tracks[trackTypes[0]]) || (state.common && state.common.track_id);
};

export const getRedirectUrl = createSelector(
    (state = {}) => state.common,
    (common = {}) => common.retpath || common.profile_url
);

export const isGDPR = createSelector(
    (state = {}) => state.settings,
    (settings = {}) => settings.tld !== 'ru' || settings.countryId !== 225 // Не Россия
);

export const isAndroidAm = createSelector(
    (state = {}) => state.am,
    (am = {}) => am.isAm && am.platform === Platforms.ANDROID
);

export const getProcessUUid = (state = {}) => {
    const {common = {}, settings = {}, auth = {}} = state;

    return common.process_uuid || settings.process_uuid || auth.process_uuid;
};

export const getUid = (state = {}) => {
    const {common = {}, auth = {}} = state;
    const {authorizedAccountsDefaultUid, defaultAccount = {}} = auth;

    return common.uid || authorizedAccountsDefaultUid || defaultAccount.uid;
};

export const isAuthRegSqSaExp = (state) => {
    const {common: {experiments: {flags = []} = {}} = {}} = state;

    return flags.includes('auth-reg-sq-sa-exp');
};

export const isLocationCountryRu = (state) => state.common.locationCountryId === 225;

const getMetricsCounterId = createSelector(
    (state = {}) => state.metrics,
    (metrics = {}) => metrics.counterId
);

export const isNeoPhonishRegisterAvailable = (state) => {
    const {
        customs: {isNeoPhonishRegisterAvailable = false} = {},
        am: {isAm = false, regType} = {},
        userEntryFlow: {allowedRegFlow = ''} = {}
    } = state;

    if (isAm) {
        return [RegType.NEOPHONISH, RegType.DOREGISH].includes(regType);
    }

    return (isNeoPhonishRegisterAvailable && allowedRegFlow !== 'portal') || allowedRegFlow === 'neophonish';
};

const SOCIALS_MAP = {
    tw: i18n('_AUTH_.twitter'),
    vk: i18n('_AUTH_.vkontakte'),
    ok: i18n('_AUTH_.odnoklassniki'),
    mr: i18n('_AUTH_.mailru'),
    gg: i18n('_AUTH_.google'),
    fb: i18n('_AUTH_.facebook'),
    apl: i18n('_AUTH_.apple'),
    esia: i18n('_AUTH_.esia')
};

export const getSocialName = (code) => SOCIALS_MAP[code];

export const getDefaultLayoutProps = (state = {}) => {
    const {
        ua,
        store = {},
        page,
        display,
        title,
        nonce,
        rumScripts,
        isLite,
        isYandex,
        isYandexInternalNetwork,
        isYandexoid,
        beatleUrl,
        opengraph,
        ysaId,
        serializedStore
    } = state;
    const {settings = {}, family: {pageInfo: {page: familyPage} = {}} = {}, common: {origin} = {}} = store;
    const language = getLanguage(store);

    return {
        page,
        familyPage,
        display,
        title,
        nonce,
        rumScripts,
        isYandex,
        isYandexInternalNetwork,
        isYandexoid,
        isYaPhoneLite: isLite,
        beatleUrl,
        ysaId,
        settings,
        language,
        origin,
        opengraph,
        version: getVersion(store),
        uatraits: ua || getUatraits(store),
        metricsUserType: getMetricsUserType(store),
        isYsaAvailable: hasExp(store, 'PASSP-22594_exp'),
        serializedStore,
        metricsCounterId: getMetricsCounterId(store)
    };
};

function validateEulaStrings(eulaStrings = {}) {
    if (!eulaStrings || typeof eulaStrings !== 'object') {
        return false;
    }

    const {
        appType,
        regFormat,
        userAgreementText,
        userAgreementUrl,
        privacyPolicyText,
        privacyPolicyUrl,
        taxiAgreementText,
        taxiAgreementUrl
    } = eulaStrings;

    return appType === 'taxi'
        ? regFormat &&
              userAgreementUrl &&
              userAgreementText &&
              privacyPolicyText &&
              privacyPolicyUrl &&
              taxiAgreementText &&
              taxiAgreementUrl
        : regFormat && userAgreementUrl && userAgreementText && privacyPolicyText && privacyPolicyUrl;
}

export const getEULATexts = createSelector(
    (state) => state.am,
    (state) => {
        const {common: {origin} = {}} = state;

        return utils.getEULATexts({isGDPR: isGDPR(state), origin});
    },
    (am = {}, eulaTexts = {}) => {
        const {eulaStrings} = am || {};

        const {title, button} = eulaTexts;

        let {description} = eulaTexts;

        if (am.isAm && eulaStrings && validateEulaStrings(eulaStrings)) {
            const {
                appType,
                regFormat,
                userAgreementText,
                userAgreementUrl,
                privacyPolicyText,
                privacyPolicyUrl,
                taxiAgreementText,
                taxiAgreementUrl
            } = eulaStrings;
            const replaceTokens = am.platform === 'android' ? ['%1$s', '%2$s', '%3$s'] : ['%@', '%@', '%@'];

            const replacers =
                appType === 'taxi'
                    ? [
                          [replaceTokens[0], userAgreementText, userAgreementUrl],
                          [replaceTokens[1], taxiAgreementText, taxiAgreementUrl],
                          [replaceTokens[2], privacyPolicyText, privacyPolicyUrl]
                      ]
                    : [
                          [replaceTokens[0], userAgreementText, userAgreementUrl],
                          [replaceTokens[1], privacyPolicyText, privacyPolicyUrl]
                      ];

            description = regFormat;

            replacers.forEach(([replaceToken, text, textUrl]) => {
                description = description.replace(replaceToken, `<a href="${textUrl}" target="_blank">${text}</a>`);
            });
        }

        return {title, description, button};
    }
);

export const isAuthComplete = (state) => state && state.common && state.common.pane === '/auth/complete';

export const isBookQREnabled = createSelector(
    (state) => state.router && state.router.location && state.router.location.pathname,
    (state) => state.auth && state.auth.isBookQREnabled,
    (pathname, isBookQREnabled) => (['/auth', '/auth/add'].includes(pathname) && isBookQREnabled) || false
);

export const isKeyEnabled2FASelector = createSelector(
    (state) => state && state.am && state.am.isAm && ['welcome', 'phoneconfirm'].includes(state.am.mode),
    (state) => state && state.customs && state.customs.isKeyEnabled2FA,
    (state) => state && state.am && state.am.enableKey2fa,
    (isCorrectMode, isKeyEnabled2FA, enableKey2faQueryParam) =>
        isCorrectMode && isKeyEnabled2FA && enableKey2faQueryParam
);
