import classnames from 'classnames';
import querystring from 'querystring';
import {errorsTxt} from '@blocks/registration/errors';
import passport from '@plibs/pclientjs';
import {DEFAULT_WEBVIEW_FINISH_URLS, WEBVIEW_STATUS_CLOSE, EULA} from './constants';

const sourceAllowedRe = /^https:\/\/(?:yandex|.+\.yandex)\.(?:ru|az|com\.am|com\.ge|co\.il|kg|lv|lt|md|tj|tm|uz|fr|ee|ua|by|kz|com|com\.tr)$/; // eslint-disable-line
const slicedOriginRe = /^https:\/\/[^/]*/;

const MONTHS = {
    '01': i18n('_AUTH_.g1'),
    '02': i18n('_AUTH_.g2'),
    '03': i18n('_AUTH_.g3'),
    '04': i18n('_AUTH_.g4'),
    '05': i18n('_AUTH_.g5'),
    '06': i18n('_AUTH_.g6'),
    '07': i18n('_AUTH_.g7'),
    '08': i18n('_AUTH_.g8'),
    '09': i18n('_AUTH_.g9'),
    10: i18n('_AUTH_.g10'),
    11: i18n('_AUTH_.g11'),
    12: i18n('_AUTH_.g12')
};

export const tldMapping = {
    com: 'com', // eslint-disable-line
    'co.il': 'com',
    fr: 'com', // eslint-disable-line
    'com.tr': 'com.tr',
    ua: 'ua' // eslint-disable-line
};

export const deepEqual = (objA, objB, ignore, level) => {
    if (objA === objB) {
        return true;
    }

    if (Object.keys && Object.keys(objA).length !== Object.keys(objB).length) {
        return false;
    }

    const isArray = Array.isArray(ignore);

    for (const key in objA) {
        if (!objA.hasOwnProperty(key)) {
            continue;
        }

        const currentLevel = !level ? key : `${level}.${key}`;

        if (ignore && ((!isArray && ignore === currentLevel) || (isArray && ignore.indexOf(currentLevel) !== -1))) {
            continue;
        }

        if (!objB.hasOwnProperty(key)) {
            return false;
        }

        if (
            objA[key] !== null &&
            objB[key] !== null &&
            typeof objA[key] === 'object' &&
            typeof objB[key] === 'object'
        ) {
            if (!deepEqual(objA[key], objB[key], ignore, currentLevel)) {
                return false;
            }
        } else if (objA[key] !== objB[key]) {
            return false;
        }
    }

    return true;
};

export default {
    getSplittedDisplayName(name) {
        if (name) {
            const match = name.match(/[^\uD800-\uDFFF]|[\uD800-\uDBFF][\uDC00-\uDFFF]|[\uD800-\uDFFF]/g);

            if (match) {
                return {
                    head: match[0],
                    tail: match.slice(1).join('')
                };
            }
        }

        return {};
    },
    dateToText(date = '0000-00-00', optins = {year: false, time: false}) {
        let {month, day, year, time, string} = '';

        if (typeof date === 'string') {
            const _date = date.split('-');

            month = MONTHS[_date[1]] || _date[1];
            day = parseInt(_date[2], 10);
            year = parseInt(_date[0], 10);
        }

        if (typeof date === 'object') {
            let _month = date.getMonth() + 1;

            _month = _month < 10 ? `0${_month}` : _month;

            month = MONTHS[_month];
            day = date.getDate();
            year = date.getYear() + 1900;

            const minutes = date.getMinutes();

            time = `${date.getHours()}:${minutes >= 10 ? minutes : `0${minutes}`}`;
        }

        string = `${day} ${month}`;

        if (optins.year) {
            string = `${string} ${year}`;
        }

        if (optins.time) {
            string = `${string}, ${time}`;
        }

        return string;
    },

    isThirdPartyUrl(url) {
        return /^(http|\/\/)/.test(url);
    }
};

export const classNames = (baseClass) => (opt, otherParams = {}) => {
    const fields = {};

    if (!opt || opt === '') {
        fields[baseClass] = true;
    } else if (typeof opt === 'string') {
        fields[`${baseClass}-${opt.toLowerCase()}`] = true;
    } else {
        Object.keys(opt).forEach((field) => {
            fields[`${baseClass}-${field.toLowerCase()}`] = opt[field];
        });
    }

    Object.keys(otherParams).forEach((field) => {
        if (otherParams[field]) {
            if (typeof otherParams[field] === 'boolean') {
                fields[`${baseClass}-${field.toLowerCase()}`] = otherParams[field];
            } else {
                fields[`${baseClass}-${field.toLowerCase()}_${otherParams[field]}`] = true;
            }
        }
    });

    return classnames(fields);
};

export const sendMessage = (msg = {}, origin = 'https://yandex.ru') => {
    const slicedOrigin = slicedOriginRe.exec(origin);

    if (process.env.NODE_ENV === 'development') {
        // eslint-disable-next-line no-console
        console.log(msg);
    }

    if (sourceAllowedRe.test(slicedOrigin)) {
        if (window.parent !== window) {
            try {
                window.parent.postMessage(msg, origin);
            } catch (e) {
                // empty
            }
        }

        if (window.opener !== window) {
            try {
                window.opener.postMessage(msg, origin);
            } catch (e) {
                // empty
            }
        }
    }
};

export const sendMessageUnsafe = (msg = {}) => {
    if (window.parent !== window) {
        try {
            window.parent.postMessage(msg, '*');
        } catch (e) {
            // empty
        }
    }
};

export const omitEmpty = (obj = {}) => {
    const result = {};

    Object.keys(obj).forEach((key) => {
        if (obj[key] !== '') {
            result[key] = obj[key];
        }
    });

    return result;
};

const PRETTIFY_PHONE_CLEAR_REGEXP = /[^\d]+/g;
const PRETTIFY_INTERNATIONAL_PHONE_FORMAT_REGEXP = /(\d{2})(\d{3})(\d{3})(\d{4})/;
const PRETTIFY_NATIONAL_PHONE_FORMAT_REGEXP = /(\d{4})(\d{3})(\d{4})/;
const PRETTIFY_RUS_PHONE_FORMAT_REGEXP = /(\d{1})(\d{3})(\d{3})(\d{2})(\d{2})/;
const PRETTIFY_RUS_PHONE_WITHOUT_PREFIX_FORMAT_REGEXP = /(\d{3})(\d{3})(\d{2})(\d{2})/;
const PRETTIFY_PHONE_PREFIX_FORMAT_REGEXP = /(\d{1})(\d{3})(\d{3})/;
const PRETTIFY_UZ_PHONE_FORMAT_REGEXP = /(\d{3})(\d{2})(\d{3})(\d{2})(\d{2})/;

export const prettifyPhone = (phone = '') => {
    let prettyPhone = phone.replace(PRETTIFY_PHONE_CLEAR_REGEXP, '');

    if (prettyPhone.length === 12) {
        prettyPhone = prettyPhone.replace(PRETTIFY_UZ_PHONE_FORMAT_REGEXP, '+$1 $2 $3-$4-$5');
    }

    if (phone.length === 12) {
        prettyPhone = prettyPhone.replace(PRETTIFY_INTERNATIONAL_PHONE_FORMAT_REGEXP, '+$1 $2 $3 $4');
    }

    if (prettyPhone.length === 11) {
        if (['7', '8'].includes(prettyPhone[0])) {
            prettyPhone = prettyPhone.replace(PRETTIFY_RUS_PHONE_FORMAT_REGEXP, '$1 ($2) $3-$4-$5');
        } else {
            prettyPhone = prettyPhone.replace(PRETTIFY_NATIONAL_PHONE_FORMAT_REGEXP, '($1) $2 $3');
        }

        if (['7', '1'].includes(prettyPhone[0])) {
            prettyPhone = `+${prettyPhone}`;
        }
    }

    if (prettyPhone.length === 10) {
        prettyPhone = prettyPhone.replace(PRETTIFY_RUS_PHONE_WITHOUT_PREFIX_FORMAT_REGEXP, '+7 ($1) $2-$3-$4');
    }

    return prettyPhone;
};

export const prettifyPhonePrefix = (phone = '') => {
    const prettyPhonePrefix = phone
        .replace(PRETTIFY_PHONE_CLEAR_REGEXP, '')
        .replace(PRETTIFY_PHONE_PREFIX_FORMAT_REGEXP, '$1 ($2) $3');

    return prettyPhonePrefix[0] === '8' ? prettyPhonePrefix : `+${prettyPhonePrefix}`;
};

export const getUnixTimeStamp = (shift = 0) => Number((new Date() / 1000 + shift) | 0);

export const getEULATexts = ({isGDPR, origin = ''}) => EULA[origin] || (isGDPR ? EULA.eulaGDPR : EULA.eulaSimple);

export const getQuery = () => querystring.parse(typeof window !== 'undefined' ? location.search.substring(1) : '');

export const getQueryParam = (name = '') => getQuery()[name];

export const getError = (field = 'global', code = '', isWarning = false) => {
    let _code = code;

    let text;

    let errorDescription;

    if (_code.indexOf(`${field}.`) === 0) {
        _code = _code.replace(`${field}.`, '');
    }

    if (errorsTxt[field] && errorsTxt[field].descriptions) {
        errorDescription = errorsTxt[field].descriptions[_code] || errorsTxt[field].descriptions[code];
    }

    if (errorsTxt[field] && errorsTxt[field].errors) {
        text = errorsTxt[field].errors[_code] || errorsTxt[field].errors[code] || errorsTxt.global.errors.global;
    }

    if (isWarning) {
        return {code: _code, text, errorDescription, warning: true};
    }

    return {code: _code, text, errorDescription};
};

export const getAdvancedError = (field = 'global', code = '') => {
    const error = getError(field, code);

    return Object.assign({}, error, {
        status: 'error',
        field
    });
};

export const removeTrailingSlash = (pathname = '') => {
    if (pathname.lastIndexOf('/') === pathname.length - 1) {
        return pathname.slice(0, -1);
    }

    return pathname;
};

export const storageAvailable = (type) => {
    let storage;

    try {
        storage = window[type];
        const x = '__storage_test__';

        storage.setItem(x, x);
        storage.removeItem(x);
        return true;
    } catch (e) {
        return (
            e instanceof DOMException &&
            // everything except Firefox
            (e.code === 22 ||
                // Firefox
                e.code === 1014 ||
                // test name field too, because code might not be present
                // everything except Firefox
                e.name === 'QuotaExceededError' ||
                // Firefox
                e.name === 'NS_ERROR_DOM_QUOTA_REACHED') &&
            // acknowledge QuotaExceededError only if there's something already stored
            storage &&
            storage.length !== 0
        );
    }
};

export function getCookie(cookieName) {
    return passport.util.getCookie(cookieName);
}

export function setCookie(cookieName, period = 7) {
    const expiresDate = new Date();

    expiresDate.setDate(expiresDate.getDate() + period);
    document.cookie = `${cookieName}=${new Date().toUTCString()};expires=${expiresDate.toUTCString()}`;
}

export const focusInput = (input) => {
    if (![HTMLInputElement, HTMLTextAreaElement].some((Element) => input instanceof Element)) {
        return;
    }

    const value = input.value;
    const range = (value || '').length * 2;

    if (range) {
        try {
            input.setSelectionRange(range, range);
        } catch (e) {
            /* EMPTY */
        }
    }

    input.focus();
    input.click();
};

export const checkDigitsAndDash = (value = '') => {
    const DIGIT_AND_DASH = /^[-0-9\s]*$/;

    return DIGIT_AND_DASH.test(value);
};

export const getFirstSymbolForQuery = (url) => {
    const queryParams = passport.util && passport.util.getUrlParamsV2(url);

    return queryParams && Object.keys(queryParams).length > 0 ? '&' : '?';
};

export const redirectToRetpathFromWebview = (retpath, status = WEBVIEW_STATUS_CLOSE) =>
    (window.location.href = retpath
        ? `${retpath}${getFirstSymbolForQuery(retpath)}status=${encodeURIComponent(status)}`
        : DEFAULT_WEBVIEW_FINISH_URLS[status]);

export function redirectToLocalUrl({pathname}) {
    window.location.assign(pathname);
}

export function redirectToExternalUrl({urlString = 'https://yandex.ru/', queryObj = {}}) {
    const form = document.createElement('form');
    const element1 = document.createElement('input');
    // compat/compat ругается, поэтому пока по старому
    // const targetUrl = new URL(urlString);
    // const query = new URLSearchParams(targetUrl.searchParams);
    // Object.entries(queryObj).forEach((pair) => query.append(...pair));
    // targetUrl.search = query;
    //
    const a = document.createElement('a');

    a.href = urlString;
    a.search = querystring.stringify(Object.assign(querystring.parse(a.search.substring(1)), queryObj));

    form.method = 'GET';
    form.action = '/redirect';

    element1.value = a.href;
    element1.name = 'url';
    form.appendChild(element1);

    document.body.appendChild(form);

    form.submit();
    form.parentNode.removeChild(form);
}

/**
 * @see https://reactjs.org/docs/legacy-event-pooling.html
 */
export function persistSyntheticEvent(func) {
    return (event) => {
        event.persist();

        return func(event);
    };
}
