$(function() {
    'use strict';
    var WAIT_TIME = 1000;

    window.YsaLogger = {
        events: [],
        timer: null,

        init: function(options) {
            if (this.initialized) {
                return;
            }

            this.initialized = true;
            this.ysaNamespace = options.ysaNamespace;
            this.ysaEsecId = options.ysaEsecId;
            this.isKeyDown = false;
            this.loggerUrl = 'ysa/event';
            this.trackUserActions();
        },
        /**
         * @private
         * @returns {undefined}
         */
        trackUserActions: function() {
            // focus/blur events don't bubble up, so we need to use focusin/focusout
            // but these events doesnt work in Firefox
            // @see https://bugzilla.mozilla.org/show_bug.cgi?id=687787
            var $document = $(document);

            if (document.addEventListener) {
                document.addEventListener('focus', this.onFocusInOut.bind(this), true);
                document.addEventListener('blur', this.onFocusInOut.bind(this), true);
            } else {
                $document.on('focusin', this.onFocusInOut.bind(this));
                $document.on('focusout', this.onFocusInOut.bind(this));
            }
            $document.on('keyup', this.onKeyDownUp.bind(this));
            $document.on('keydown', this.onKeyDownUp.bind(this));
        },
        /**
         * @private
         * @param {FocusEvent} event - focus/blur/focusin/focusout event
         * @returns {undefined}
         * Listen to focus/blur events and log field name and its value length
         */
        onFocusInOut: function(event) {
            var field = event.target;
            var isPasswordField = field.name === 'password' || field.type === 'password';
            var eventType = event.type.match(/^focus$|^focusin$/) ? 'focus' : 'blur';

            if (event.target.tagName === 'INPUT') {
                this.submitEvent({
                    name: eventType,
                    ts: new Date().valueOf() / 1000,
                    payload: {
                        field_name: field.name,
                        content: isPasswordField ? '' : field.value
                    }
                });
            }
        },
        /**
         * Listen to keydown/up event and log pressed keys with modificators
         * logging is disabled for password fields
         * @private
         * @param {KeyboardEvent} event - keydown/keyup events
         * @return {undefined}
         */
        onKeyDownUp: function(event) {
            var pressedModificators = [];
            var modificators = ['alt', 'ctrl', 'shift', 'meta'];

            if (
                event.target.type === 'password' ||
                event.target.name === 'password' ||
                (event.type === 'keydown' && this.isKeyDown)
            ) {
                return;
            }
            if (event.type === 'keydown') {
                this.isKeyDown = true;
            } else {
                this.isKeyDown = false;
            }

            modificators.forEach(function(mod) {
                if (event[mod + 'Key']) {
                    pressedModificators.push(mod);
                }
            });
            this.submitEvent({
                name: event.type,
                ts: new Date().valueOf() / 1000,
                payload: {
                    key: event.key,
                    key_code: event.keyCode,
                    modificators: pressedModificators
                }
            });
        },
        /**
         * @param {Event} event - submit event
         * @returns {undefined}
         */
        submitEvent: function(event) {
            var self = this;

            if (!this.initialized) {
                return;
            }

            clearTimeout(this.timer);

            this.events.push(event);
            this.timer = setTimeout(function() {
                passport.api.request(self.loggerUrl, {
                    ysaNamespace: self.ysaNamespace,
                    ysaEsecId: self.ysaEsecId,
                    events: JSON.stringify(self.events)
                });

                self.events = [];
            }, WAIT_TIME);
        },

        flushAllEvents: function() {
            clearTimeout(this.timer);

            if (this.initialized && this.events.length) {
                passport.api.request(this.loggerUrl, {
                    ysaNamespace: this.ysaNamespace,
                    ysaEsecId: this.ysaEsecId,
                    events: JSON.stringify(this.events)
                });
                this.events = [];
            }
        }
    };
});
